<?php declare(strict_types=1);

/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Connect
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Frakmenta\WooCommerce\Utils;

use WC_Log_Handler_File;

/**
 * Class Logger
 *
 * @see https://tools.ietf.org/html/rfc5424#page-11
 *
 * @package Frakmenta\WooCommerce\Utils
 * @since    4.4.2
 */
class Logger {

    /**
     * Check if debug mode is enabled
     * 
     * @return bool
     */
    private static function is_debug_enabled(): bool {
        // Check WP_DEBUG
        if (defined('WP_DEBUG') && WP_DEBUG) {
            return true;
        }
        
        // Check WC logging level
        if (defined('WC_LOG_THRESHOLD') && WC_LOG_THRESHOLD !== '') {
            return true;
        }
        
        // Check Frakmenta debug option
        if (get_option('FRAKMENTA_DEBUG', false)) {
            return true;
        }
        
        return false;
    }

    /**
     * Write directly to PHP error log (always enabled)
     * 
     * @param string $level
     * @param string $message
     * @return void
     */
    private static function write_log(string $level, string $message): void {
        // Only write debug and info logs if debug is enabled
        if (in_array($level, ['debug', 'info']) && !self::is_debug_enabled()) {
            return;
        }
        
        $timestamp = date('Y-m-d H:i:s');
        $formatted_message = "[{$timestamp}] FRAKMENTA [{$level}] {$message}";
        error_log($formatted_message);
        
        // Also try WC logger if available
        if (function_exists('wc_get_logger')) {
            try {
                $logger = wc_get_logger();
                $logger->log($level, $message, array('source' => 'frakmenta'));
            } catch (\Exception $e) {
                // Ignore WC logger errors
            }
        }
    }

    /**
     * Log method for emergency level
     *
     * System is unusable. This is the highest severity level.
     * Use only for catastrophic failures that require immediate attention.
     *
     * @since  4.4.2
     * @param  string $message Error message describing the emergency
     * @return void
     *
     * @example
     * Logger::log_emergency('Database connection completely failed');
     */
    public static function log_emergency(string $message): void {
        $logger = wc_get_logger();
        $logger->log('emergency', $message, array('source' => 'frakmenta'));
    }

    /**
     * Log method for alert level
     *
     * Action must be taken immediately.
     * Use for critical conditions that require immediate intervention.
     *
     * @since  4.4.2
     * @param  string $message Error message describing the alert
     * @return void
     *
     * @example
     * Logger::log_alert('Entire website down, database unavailable');
     * Logger::log_alert('Payment gateway completely offline');
     */
    public static function log_alert(string $message): void {
        $logger = wc_get_logger();
        $logger->log('alert', $message, array('source' => 'frakmenta'));
    }

    /**
     * Log method for critical level
     *
     * Critical conditions that don't require immediate action but need urgent attention.
     * Use for unexpected exceptions and severe errors.
     *
     * @since  4.4.2
     * @param  string $message Error message describing the critical condition
     * @return void
     *
     * @example
     * Logger::log_critical('Unexpected exception in payment processing');
     * Logger::log_critical('API authentication failed permanently');
     */
    public static function log_critical(string $message): void {
        $logger = wc_get_logger();
        $logger->log('critical', $message, array('source' => 'frakmenta'));
    }

    /**
     * Log method for error level
     *
     * Runtime errors that don't require immediate action but should be logged and monitored.
     * Use for recoverable errors and failed operations.
     *
     * @since  4.4.2
     * @param  string $message Error message describing the error condition
     * @return void
     *
     * @example
     * Logger::log_error('Failed to create Frakmenta operation: Invalid amount');
     * Logger::log_error('API request timeout after 3 retries');
     */
    public static function log_error(string $message): void {
        self::write_log('error', $message);
    }

    /**
     * Log method for warning level
     *
     * Exceptional occurrences that are not errors.
     * Use for deprecated features, poor use of APIs, or things that may become errors.
     *
     * @since  4.4.2
     * @param  string $message Warning message
     * @return void
     *
     * @example
     * Logger::log_warning('Using deprecated API endpoint, will be removed in v3.0');
     * Logger::log_warning('API response time exceeded 2 seconds');
     */
    public static function log_warning(string $message): void {
        self::write_log('warning', $message);
    }

    /**
     * Log method for notice level
     *
     * Normal but significant events that should be noted.
     * Use for important business logic events.
     *
     * @since  4.4.2
     * @param  string $message Notice message
     * @return void
     *
     * @example
     * Logger::log_notice('Payment processed with different method than registered');
     * Logger::log_notice('Order status changed from pending to processing');
     */
    public static function log_notice(string $message): void {
        $logger = wc_get_logger();
        $logger->log('notice', $message, array('source' => 'frakmenta'));
    }

    /**
     * Log method for info level
     *
     * Interesting informational events.
     * Only logs when debug mode is enabled.
     *
     * @since  4.4.2
     * @param  string $message Informational message
     * @return void
     *
     * @example
     * Logger::log_info('Payment link generated: https://frakmenta.com/op/...');
     * Logger::log_info('API request successful, operation ID: 123456');
     */
    public static function log_info(string $message): void {
        self::write_log('info', $message);
    }

    /**
     * Log method for debug level
     *
     * Detailed debug information for development.
     * Only logs when debug mode is enabled.
     *
     * @since  4.4.2
     * @param  string $message Debug message with detailed information
     * @return void
     *
     * @example
     * Logger::log_debug('Request payload: ' . json_encode($data));
     * Logger::log_debug('Order processing step 3 of 5 completed');
     */
    public static function log_debug(string $message): void {
        self::write_log('debug', $message);
    }

    /**
     * Return an array of logs filenames
     *
     * @return array
     */
    private function get_logs(): array {
        $logs = WC_Log_Handler_File::get_log_files();
        return $logs;
    }

    /**
     * Get all log files that belong to Frakmenta
     *
     * Filters WooCommerce log files to return only those containing 'frakmenta' in the filename.
     * Used in admin panel to display Frakmenta-specific logs.
     *
     * @since  4.4.2
     * @return array<string, string> Array of log files with filename as key and full name as value
     *
     * @example
     * $logger = new Logger();
     * $logs = $logger->get_Frakmenta_logs();
     * // Returns: ['frakmenta-2025-11-28-abc123' => 'frakmenta-2025-11-28-abc123.log', ...]
     */
    public function get_Frakmenta_logs(): array {
        $logs = $this->get_logs();
        foreach ($logs as $key => $log) {
            if (strpos($log, 'frakmenta') === false) {
                unset($logs[$key]);
            }
        }
        return $logs;
    }

}
