<?php declare(strict_types=1);

/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Connect
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Frakmenta\WooCommerce\Utils;

/**
 * Frakmenta Common Utilities
 *
 * Provides common utility methods for the Frakmenta plugin.
 *
 * @package Frakmenta\WooCommerce\Utils
 * @since   2.0.0
 */
class FrakmentaCommonUtils
{
    /**
     * Get current date in Y-m-d format
     *
     * Returns the current date in ISO 8601 format (Y-m-d) using UTC timezone.
     * Used for date fields in Frakmenta API requests.
     *
     * @since  1.0.0
     * @return string Current date in Y-m-d format (e.g., '2025-11-28')
     */
    public static function get_current_date(): string {
        $defaultTimeZone = 'UTC';
        return date('Y-m-d');
    }

    /**
     * Get all current Frakmenta configuration parameters
     *
     * Retrieves all Frakmenta plugin configuration options from WordPress options table.
     * This includes API credentials, mode settings, and feature flags.
     *
     * @since  1.0.0
     * @return array<string, mixed> Associative array with all Frakmenta configuration parameters:
     *                              - FRAKMENTA_DELEGATION: Delegation ID
     *                              - FRAKMENTA_EXIST_ACCOUNT: Whether account exists (0/1)
     *                              - FRAKMENTA_URL: API base URL
     *                              - FRAKMENTA_PUBLIC_KEY: Public API key
     *                              - FRAKMENTA_MERCHANT_ID: Merchant identifier
     *                              - FRAKMENTA_MODE: Operating mode (0=test, 1=production)
     *                              - FRAKMENTA_PRODUCT_OPTION: Product widget option (0/1)
     *                              - LOCATION_SIMULATOR_DEFAULT: Widget location selector
     *                              - FRAKMENTA_PRIVATE_KEY: Private API key for signatures
     */
    public static function get_frakmenta_current_parameters(): array {
        return array(
            'FRAKMENTA_DELEGATION' => get_option('FRAKMENTA_DELEGATION'),
            'FRAKMENTA_EXIST_ACCOUNT' => get_option('FRAKMENTA_EXIST_ACCOUNT'),
            'FRAKMENTA_URL' => get_option('FRAKMENTA_URL'),
            'FRAKMENTA_PUBLIC_KEY' => get_option('FRAKMENTA_PUBLIC_KEY'),
            'FRAKMENTA_MERCHANT_ID' => get_option('FRAKMENTA_MERCHANT_ID'),
            'FRAKMENTA_MODE' => get_option('FRAKMENTA_MODE'),
            'FRAKMENTA_PRODUCT_OPTION' => get_option('FRAKMENTA_PRODUCT_OPTION'),
            'LOCATION_SIMULATOR_DEFAULT' => get_option('LOCATION_SIMULATOR_DEFAULT'),
            'FRAKMENTA_PRIVATE_KEY' => get_option('FRAKMENTA_PRIVATE_KEY')
        );
    }

    /**
     * Clean Frakmenta operation ID by removing prefix
     *
     * Removes the 'frakmenta-' prefix (10 characters) from the transaction ID
     * stored in WooCommerce orders to get the actual Frakmenta operation ID.
     *
     * @since  1.0.0
     * @param  string $id Full transaction ID with prefix (e.g., 'frakmenta-123456')
     * @return string     Clean operation ID without prefix (e.g., '123456')
     *
     * @example
     * FrakmentaCommonUtils::clean_operation_id_frakmenta('frakmenta-789012');
     * // Returns: '789012'
     */
    public static function clean_operation_id_frakmenta(string $id): string {
        return substr($id, 10);
    }
}