<?php declare(strict_types=1);

/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Settings
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Frakmenta\WooCommerce\Settings;

/**
 * The settings page controller.
 *
 * Defines all the functionalities needed on the settings page
 *
 * @since   4.0.0
 */
class SettingsController {

    /**
     * In plugin version < 4.0.0 the options frakmenta_testmode, and frakmenta_debugmode
     * had been stored as strings and returns yes - no.
     *
     * This function also works to returns booleans instead of strings for
     * frakmenta_second_chance and frakmenta_remove_all_settings options
     *
     * @since 4.0.0
     * @see https://developer.wordpress.org/reference/hooks/option_option/
     *
     * @param   string $value
     * @return  boolean
     */
    public function filter_frakmenta_settings_as_booleans( string $value ): bool {
        if ( 'yes' === $value || '1' === $value ) {
            return true;
        }
        return false;
    }

    /**
     * This function returns int instead of strings for frakmenta_time_active
     *
     * @see https://developer.wordpress.org/reference/hooks/option_option/
     *
     * @param   string $value
     * @return  integer
     */
    public function filter_frakmenta_settings_as_int( string $value ): int {
        return (int) $value;
    }

    /**
     * Register the stylesheets for the settings page.
     *
     * @see https://developer.wordpress.org/reference/functions/wp_enqueue_style/
     * @return void
     */
    public function enqueue_styles(): void {
        // Only load on Frakmenta settings page
        $screen = get_current_screen();
        if ( ! $screen || strpos( $screen->id, 'frakmenta-settings' ) === false ) {
            return;
        }

        // Load Bootstrap components and UI framework styles
        wp_enqueue_style( 
            'frakmenta-additionals-css', 
            FRAKMENTA_PLUGIN_URL . '/assets/commons/css/frakmenta_additionals.css', 
            array(), 
            FRAKMENTA_PLUGIN_VERSION, 
            'all' 
        );

        // Load custom Frakmenta admin styles
        wp_enqueue_style( 
            'frakmenta-style-css', 
            FRAKMENTA_PLUGIN_URL . '/assets/commons/css/frakmenta_style.css', 
            array( 'frakmenta-additionals-css' ), 
            FRAKMENTA_PLUGIN_VERSION, 
            'all' 
        );

        // Load main Frakmenta CSS (must load last for proper cascading)
        wp_enqueue_style( 
            'frakmenta-commons-css', 
            FRAKMENTA_PLUGIN_URL . '/assets/commons/css/frakmenta.css', 
            array( 'frakmenta-additionals-css', 'frakmenta-style-css' ), 
            FRAKMENTA_PLUGIN_VERSION, 
            'all' 
        );
    }

    /**
     * Register the JavaScript needed in the backend.
     *
     * @see https://developer.wordpress.org/reference/functions/wp_enqueue_script/
     * @see https://developer.wordpress.org/reference/functions/wp_localize_script/
     * @return void
     */
    public function enqueue_scripts():void {
        // Only load on Frakmenta settings page
        $screen = get_current_screen();
        if ( ! $screen || strpos( $screen->id, 'frakmenta-settings' ) === false ) {
            return;
        }

        self::default_values_plugin();

        $frakmenta_vars = array(
                    "FRAKMENTA_DELEGATION" => get_option('FRAKMENTA_DELEGATION'),
                    "FRAKMENTA_EXIST_ACCOUNT" => get_option('FRAKMENTA_EXIST_ACCOUNT'),
                    "FRAKMENTA_URL" => get_option('FRAKMENTA_URL'),
                    "FRAKMENTA_PUBLIC_KEY" => get_option('FRAKMENTA_PUBLIC_KEY'),
                    "FRAKMENTA_MERCHANT_ID" => get_option('FRAKMENTA_MERCHANT_ID'),
                    "FRAKMENTA_MODE" => get_option('FRAKMENTA_TEST_MODE'),
                    "FRAKMENTA_PRODUCT_OPTION" => get_option('FRAKMENTA_PRODUCT_OPTION'),
                    "LOCATION_SIMULATOR_DEFAULT" => get_option('LOCATION_SIMULATOR_DEFAULT')
        );

        wp_register_script( 'frakmenta-admin-js', FRAKMENTA_PLUGIN_URL . '/assets/admin/js/frakmenta-admin.js', array( 'jquery' ), FRAKMENTA_PLUGIN_VERSION, false );

        wp_localize_script( 'frakmenta-admin-js', 'frakmenta', $frakmenta_vars );
        wp_enqueue_script( 'frakmenta-admin-js' );
    }

    /**
     * Register the common settings page in WooCommerce menu section.
     *
     * @see https://developer.wordpress.org/reference/functions/add_submenu_page/
     * @return void
     */
    public function register_common_settings_page(): void {
        $title = sprintf( __( 'Frakmenta Configuración v. %s', 'frakmenta' ), FRAKMENTA_PLUGIN_VERSION ); // phpcs:ignore WordPress.WP.I18n.MissingTranslatorsComment
        add_submenu_page(
            'woocommerce',
            esc_html( $title ),
            __( 'Configuración Frakmenta', 'frakmenta' ),
            'manage_woocommerce',
            'frakmenta-settings',
            array( $this, 'display_frakmenta_settings' )
        );
    }

    /**
     * Returns active tab defined in get variable
     *
     * @return  string
     */
    private function get_tab_active(): string {
        if ( ! isset( $_GET['tab'] ) || '' === $_GET['tab'] ) {
            $tab_active = 'general';
        }
        if ( isset( $_GET['tab'] ) && '' !== $_GET['tab'] ) {
            $tab_active = $_GET['tab'];
        }
        return $tab_active;
    }

    /**
     * Register general settings in common settings page
     *
     * @return void
     */
    public function register_common_settings(): void {
        $settings_fields = new SettingsFields();
        $settings        = $settings_fields->get_settings();
        foreach ( $settings as $tab_key => $section ) {
            $this->add_settings_section( $tab_key, $section['title'] );
            foreach ( $section['fields'] as $field ) {
                $this->register_setting( $field, $tab_key );
                $this->add_settings_field( $field, $tab_key );
            }
        }
    }

    /**
     * Add settings field
     *
     * @see https://developer.wordpress.org/reference/functions/add_settings_field/
     * @param   array  $field      The field
     * @param   string $tab_key    The key of the tab
     * @return  void
     */
    private function add_settings_field( array $field, string $tab_key ): void {
        add_settings_field(
            $field['id'],
            $this->generate_label_for_settings_field( $field ),
            array( $this, 'display_field' ),
            'frakmenta-settings-' . $tab_key,
            $tab_key,
            array(
                'field' => $field,
            )
        );
    }

    /**
     * Return the label tag to be used in add_settings_field
     *
     * @param   array $field  The settings field array
     * @return  string
     */
    private function generate_label_for_settings_field( array $field ): string {
        if ( '' === $field['tooltip'] ) {
            return sprintf( '<label for="%s">%s</label>', $field['id'], $field['label'] );
        }
        return sprintf( '<label for="%s">%s %s</label>', $field['id'], $field['label'], wc_help_tip( $field['tooltip'] ) );
    }

    /**
     * Filter which set the settings page and adds a screen options of WooCommerce
     *
     * @see http://hookr.io/filters/woocommerce_screen_ids/
     * @param   array $screen
     * @return  array
     */
    public function set_wc_screen_options_in_common_settings_page( array $screen ): array {
        $screen[] = 'woocommerce_page_frakmenta-settings';
        return $screen;
    }

    /**
     * Register setting
     *
     * @see https://developer.wordpress.org/reference/functions/register_setting/
     * @param   array  $field
     * @param   string $tab_key
     * @return  void
     */
    private function register_setting( array $field, string $tab_key ): void {
        register_setting(
            'frakmenta-settings-' . $tab_key,
            $field['id'],
            array(
                'type'              => $field['setting_type'],
                'show_in_rest'      => false,
                'sanitize_callback' => $field['callback'],
            )
        );
    }

    /**
     * Add settings section
     *
     * @see https://developer.wordpress.org/reference/functions/add_settings_section/
     *
     * @param   string $section_key
     * @param   string $section_title
     * @return  void
     */
    private function add_settings_section( string $section_key, string $section_title ): void {
        add_settings_section(
            $section_key,
            $section_title,
            array( $this, 'display_intro_section' ),
            'frakmenta-settings-' . $section_key
        );
    }

    /**
     * Callback to display the title on each settings sections
     *
     * @see https://developer.wordpress.org/reference/functions/add_settings_section/
     * @param   array $args
     * @return  void
     */
    public function display_intro_section( array $args ): void {
        $settings_fields = new SettingsFields();
        $settings        = $settings_fields->get_settings();
        if ( ! empty( $settings[ $args['id'] ]['intro'] ) ) {
            esc_html( printf( '<p>%s</p>', $settings[ $args['id'] ]['intro'] ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        }
    }

    /**
     * Return the HTML view by field.
     * Is the callback function in add_settings_field
     *
     * @param   array $args
     * @return  void
     */
    public function display_field( $args ): void {
        $field                  = $args['field'];
        $settings_field_display = new SettingsFieldsDisplay( $field );
        $settings_field_display->display();
    }

    /**
     * Display the common settings page view.
     *
     * @return void
     */
    public function display_frakmenta_settings(): void {

        $tab_active   = $this->get_tab_active();
        /*$needs_update = $this->needs_update();*/
        remove_query_arg( 'needs-setup' );
        require_once FRAKMENTA_PLUGIN_DIR_PATH . 'templates/fk_admin_page.php';
    }

    /**
     * Set Frakmenta default parameters on first activation
     *
     * Initializes default configuration values for Frakmenta plugin if they don't exist.
     * This method is called on plugin activation and admin page load.
     * The user must configure the plugin before it becomes active.
     *
     * Default values set:
     * - FRAKMENTA_DELEGATION: 1
     * - FRAKMENTA_EXIST_ACCOUNT: 0 (not configured yet)
     * - FRAKMENTA_LOCATION_SIMULATOR: '.product_attributes'
     * - FRAKMENTA_PRODUCT_OPTION: 0 (widget disabled)
     *
     * @since  1.0.0
     * @return void
     */
    public function default_values_plugin(): void {
        // Only initialize if not already configured
        if (empty(get_option('FRAKMENTA_DELEGATION'))) {
            update_option('FRAKMENTA_DELEGATION', 1);
            update_option('FRAKMENTA_EXIST_ACCOUNT', 0);
            update_option('FRAKMENTA_LOCATION_SIMULATOR', '.product_attributes');
            update_option('FRAKMENTA_PRODUCT_OPTION', 0);
        }
    }

    /**
     * Process admin configuration form submission
     *
     * Handles the submission of the Frakmenta configuration form from admin panel.
     * Updates all plugin options based on form data, including API credentials and mode.
     *
     * Depending on the mode selected:
     * - Production mode (1): Uses credentials from form
     * - Test mode (0): Uses predefined beta environment credentials
     *
     * Security: Should verify nonce and capabilities before processing in production.
     *
     * @since  1.0.0
     * @return void
     *
     * @global array $_POST Form data containing:
     *                      - submitButton: Presence indicates form submission
     *                      - fk_account: Account existence flag
     *                      - fk_sim_product: Product widget option
     *                      - fk_location_simulator: Widget location selector
     *                      - fk_mode: Operating mode (0=test, 1=production)
     *                      - fk_public_key: Public API key (production only)
     *                      - fk_merchant_id: Merchant ID (production only)
     *                      - fk_private_key: Private API key (production only)
     *
     * @todo Add nonce verification for security
     * @todo Add capability check (manage_woocommerce)
     * @todo Sanitize all POST inputs
     */
    public function process_admin_form(): void {
        if (!empty($_POST['submitButton'])) {
            // Sanitize inputs
            $fk_account = sanitize_text_field(wp_unslash($_POST['fk_account'] ?? '0'));
            $fk_sim_product = sanitize_text_field(wp_unslash($_POST['fk_sim_product'] ?? '0'));
            $fk_location_simulator = sanitize_text_field(wp_unslash($_POST['fk_location_simulator'] ?? '.product-add-to-cart'));
            $fk_mode = sanitize_text_field(wp_unslash($_POST['fk_mode'] ?? '0'));
            
            update_option('FRAKMENTA_EXIST_ACCOUNT', $fk_account);
            update_option('FRAKMENTA_PRODUCT_OPTION', $fk_sim_product);
            update_option('FRAKMENTA_LOCATION_SIMULATOR', $fk_location_simulator);
            update_option('FRAKMENTA_MODE', $fk_mode);

            if ($fk_mode == '1') {
                // Production mode - use provided credentials
                $fk_public_key = sanitize_text_field(wp_unslash($_POST['fk_public_key'] ?? ''));
                $fk_merchant_id = sanitize_text_field(wp_unslash($_POST['fk_merchant_id'] ?? ''));
                $fk_private_key = sanitize_text_field(wp_unslash($_POST['fk_private_key'] ?? ''));
                
                update_option('FRAKMENTA_URL', 'https://frakmenta.com');
                update_option('FRAKMENTA_PUBLIC_KEY', $fk_public_key);
                update_option('FRAKMENTA_MERCHANT_ID', $fk_merchant_id);
                update_option('FRAKMENTA_PRIVATE_KEY', $fk_private_key);
            } else {
                // Test mode - use beta credentials
                update_option('FRAKMENTA_URL', 'https://beta2.frakmenta.com');
                update_option('FRAKMENTA_PUBLIC_KEY', '91bdde4687b07c34a8af8e23c0f6b1a8dfa1a5c72517fb55116c11741fd98a74');
                update_option('FRAKMENTA_MERCHANT_ID', '30100');
                update_option('FRAKMENTA_PRIVATE_KEY', 'eb73e21d88eaedab5cf1d3a6a6b558fe7917204e86120120f0f6cf7ea49ebbdc');
            }
            
            // Redirect with success message
            wp_redirect(admin_url('admin.php?page=frakmenta-settings&settings-updated=1'));
            exit;
        }
    }

    /**
     * Get Frakmenta admin parameters as JSON
     *
     * Returns all Frakmenta configuration parameters as a JSON-encoded string.
     * Used for JavaScript integration in admin pages.
     *
     * @since  1.0.0
     * @return string JSON-encoded array of Frakmenta parameters
     *
     * @see self::default_values_plugin() Called to ensure defaults exist
     *
     * @deprecated Consider using FrakmentaCommonUtils::get_frakmenta_current_parameters() instead
     */
    public function fk_admin_parameters(): string {
        // Ensure default configuration exists
        frakmenta_default_configuration();
        
        return json_encode(array(
            'FRAKMENTA_DELEGATION' => get_option('FRAKMENTA_DELEGATION'),
            'FRAKMENTA_EXIST_ACCOUNT' => get_option('FRAKMENTA_EXIST_ACCOUNT'),
            'FRAKMENTA_URL' => get_option('FRAKMENTA_URL'),
            'FRAKMENTA_PUBLIC_KEY' => get_option('FRAKMENTA_PUBLIC_KEY'),
            'FRAKMENTA_MERCHANT_ID' => get_option('FRAKMENTA_MERCHANT_ID'),
            'FRAKMENTA_MODE' => get_option('FRAKMENTA_TEST_MODE'),
            'FRAKMENTA_PRODUCT_OPTION' => get_option('FRAKMENTA_PRODUCT_OPTION'),
            'LOCATION_SIMULATOR_DEFAULT' => get_option('LOCATION_SIMULATOR_DEFAULT')
        ));
    }

    /**
     * Add settings link to plugin actions in plugins list
     *
     * @param array $links Existing plugin action links
     * @return array Modified plugin action links with settings link added
     */
    public function add_settings_link( array $links ): array {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=frakmenta-settings'),
            __('Ajustes', 'frakmenta')
        );
        
        // Add the settings link at the beginning of the array
        array_unshift($links, $settings_link);
        
        return $links;
    }

}
