<?php declare(strict_types=1);

/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Payments
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Frakmenta\WooCommerce\Services;

use WC_Order;
use Frakmenta\WooCommerce\Utils\Logger;
use Frakmenta\WooCommerce\Utils\FrakmentaCommonUtils;
use Frakmenta\WooCommerce\Services\APIService;
/**
 * Class OrderService
 *
 * @package Frakmenta\WooCommerce\Services
 */
class OrderService {


    /**
     * @var CustomerService
     */
    private $customer_service;

    /**
     * @var ShoppingCartService
     */
    private $shopping_cart_service;

    /**
     * @var connectApi
     */
    private $connectApi;

    /**
     * OrderService constructor.
     */
    public function __construct() {
        $this->connectApi = new APIService();
    }

    /**
     * Create order in Frakmenta
     *
     * @param WC_Order $order
     * @return string Payment URL
     */
    public function create_order_frakmenta(WC_Order $order): string {
        $cart_products = $this->buildCartProducts($order);
        $customer = $this->buildCustomerData($order);
        $orderShop = $this->buildOrderData($order, $cart_products);
        $frakmenta_parameters = FrakmentaCommonUtils::get_frakmenta_current_parameters();
        
        $flowConfig = $this->buildFlowConfig($order);
        $invoice_id = $this->generateInvoiceId($frakmenta_parameters['FRAKMENTA_MERCHANT_ID'], $order->get_id());
        $product_price = $this->calculateProductPrice((float) $order->get_total());

        $data = $this->buildOperationData(
            $frakmenta_parameters,
            $invoice_id,
            $product_price,
            $customer,
            $orderShop,
            $flowConfig
        );

        $signature = $this->generateSignature($data, $frakmenta_parameters['FRAKMENTA_PRIVATE_KEY']);
        $data['signature'] = $signature;

        Logger::log_info('╔══════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   ENVIANDO PETICIÓN A FRAKMENTA API /operations   ║');
        Logger::log_info('╚══════════════════════════════════════════════════════════════╝');
        Logger::log_info('URL: ' . $frakmenta_parameters['FRAKMENTA_URL'] . '/api/fk/v2/operations');
        Logger::log_info('JSON Completo enviado:');
        Logger::log_info(json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

        $response = $this->connectApi->connection(
            $frakmenta_parameters['FRAKMENTA_URL'] . '/api/fk/v2/operations',
            json_encode($data),
            'POST'
        );

        Logger::log_info('╔══════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   RESPUESTA DE FRAKMENTA API /operations   ║');
        Logger::log_info('╚══════════════════════════════════════════════════════════════╝');
        Logger::log_info('Respuesta completa:');
        Logger::log_info($response);

        $response_frakmenta = json_decode($response);

        if (!isset($response_frakmenta->status) || strtoupper($response_frakmenta->status) !== 'OK') {
            Logger::log_error('Error en la operación de Frakmenta: ' . json_encode($data));
            throw new \Exception('Error al crear operación en Frakmenta');
        }

        $order->set_transaction_id('frakmenta-' . $response_frakmenta->data->operation_id);
        $order->save();

        return $frakmenta_parameters['FRAKMENTA_URL'] . '/op/ecommerce/' . $response_frakmenta->data->token_url . '/load';
    }

    /**
     * Build cart products array from order items
     *
     * @param WC_Order $order
     * @return array
     */
    private function buildCartProducts(WC_Order $order): array {
        $cart_products = [];

        foreach ($order->get_items() as $item_id => $item) {
            $link = 'https://frakmenta.com/no-link';
            
            if ($item->get_type() === 'line_item') {
                $product = $item->get_product();
                if ($product) {
                    $link = get_permalink($product->get_id());
                }
            }

            $cart_products[] = [
                'id' => strval($item->get_product_id()),
                'name' => substr($item->get_name(), 0, 900),
                'quantity' => $item->get_quantity(),
                'price' => number_format(floatval($item->get_total()), 2, '.', ''),
                'tax_rate' => number_format(floatval($item->get_total_tax()), 2, '.', ''),
                'description' => $item->get_name(),
                'url' => $link,
                'image_url' => $link
            ];
        }

        return $cart_products;
    }

    /**
     * Build customer data array
     *
     * @param WC_Order $order
     * @return array
     */
    private function buildCustomerData(WC_Order $order): array {
        return [
            'identification' => [
                'nif' => ' ',
                'legal_first_name' => $order->get_billing_first_name() ?: ' ',
                'legal_last_name' => $order->get_billing_last_name() ?: ' ',
                'date_of_birth' => FrakmentaCommonUtils::get_current_date(),
                'mobile_phone_number' => $order->get_billing_phone() ?: ' ',
                'email' => $order->get_billing_email()
            ],
            'address' => [
                'line_1' => $order->get_billing_address_1(),
                'line_2' => $order->get_billing_address_2() ?: ' ',
                'phone' => $order->get_billing_phone() ?: ' ',
                'city' => $order->get_billing_city(),
                'state' => $order->get_billing_state(),
                'county' => ',',
                'country_code' => $order->get_billing_country(),
                'postcode' => $order->get_billing_postcode(),
            ],
            'store_details' => [
                'customer_date_joined' => FrakmentaCommonUtils::get_current_date(),
                'customer_last_login' => FrakmentaCommonUtils::get_current_date(),
            ],
            'financial' => [
                'salary' => 0,
                'currency' => 'EUR',
                'employment_status' => 'N/A',
                'contract_type' => 'N/A'
            ],
            'other_data' => [
                ['name' => 'Tienda', 'type' => 'STRING', 'value' => get_bloginfo('name')],
                ['name' => 'Ecommerce', 'type' => 'STRING', 'value' => 'WOOCOMMERCE'],
                ['name' => 'Version', 'type' => 'STRING', 'value' => FRAKMENTA_PLUGIN_VERSION],
                ['name' => 'Environment', 'type' => 'STRING', 'value' => get_option('FRAKMENTA_MODE') == 0 ? 'TEST' : 'PRODUCTION']
            ]
        ];
    }

    /**
     * Build order data array
     *
     * @param WC_Order $order
     * @param array $cart_products
     * @return array
     */
    private function buildOrderData(WC_Order $order, array $cart_products): array {
        return [
            'id' => strval($order->get_id()),
            'products' => $cart_products
        ];
    }

    /**
     * Build flow configuration
     *
     * @param WC_Order $order
     * @return array
     */
    private function buildFlowConfig(WC_Order $order): array {
        $order_received_url = $order->get_checkout_order_received_url();
        $order_checkout_url = wc_get_checkout_url();

        return [
            'success_url' => $order_received_url,
            'notification_url' => 'https://frakmenta.com',
            'ko_url' => $order_checkout_url,
        ];
    }

    /**
     * Generate unique invoice ID
     *
     * @param string $merchant_id
     * @param int $order_id
     * @return string
     */
    private function generateInvoiceId(string $merchant_id, int $order_id): string {
        return hash('sha256', $merchant_id . '-' . $order_id . '-' . date('YmdHis'));
    }

    /**
     * Calculate product price in cents
     *
     * @param float $total
     * @return int
     */
    private function calculateProductPrice(float $total): int {
        return (int) (number_format($total, 2, '.', '') * 100);
    }

    /**
     * Build operation data array
     *
     * @param array $frakmenta_parameters
     * @param string $invoice_id
     * @param int $product_price
     * @param array $customer
     * @param array $orderShop
     * @param array $flowConfig
     * @return array
     */
    private function buildOperationData(
        array $frakmenta_parameters,
        string $invoice_id,
        int $product_price,
        array $customer,
        array $orderShop,
        array $flowConfig
    ): array {
        return [
            'merchant_id' => $frakmenta_parameters['FRAKMENTA_MERCHANT_ID'],
            'invoice_id' => $invoice_id,
            'product_price' => $product_price,
            'currency_code' => 'EUR',
            'delegation' => '1',
            'type' => 'e-commerce',
            'customer' => $customer,
            'order' => $orderShop,
            'flow_config' => $flowConfig,
            'other_data' => [['name' => 'N/A', 'type' => 'STRING', 'value' => 'N/A']]
        ];
    }

    /**
     * Generate signature for operation
     *
     * @param array $data
     * @param string $private_key
     * @return string
     */
    private function generateSignature(array $data, string $private_key): string {
        return hash(
            'sha256',
            $data['merchant_id'] . '|' .
            $data['delegation'] . '|' .
            $data['type'] . '|' .
            $data['invoice_id'] . '|' .
            $data['product_price'] . '|' .
            $data['currency_code'] . '|' .
            $private_key,
            false
        );
    }

    /**
     * Get status of Frakmenta operation
     *
     * @param WC_Order $order
     * @return string 'success' or 'error'
     */
    public function getStatusOperationFrakmenta(WC_Order $order): string {
        Logger::log_info('╔══════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   VERIFICANDO ESTADO DE OPERACIÓN FRAKMENTA   ║');
        Logger::log_info('╚══════════════════════════════════════════════════════════════╝');
        
        $frakmenta_parameters = FrakmentaCommonUtils::get_frakmenta_current_parameters();
        $transaction_id = $order->get_transaction_id();
        Logger::log_info('Transaction ID completo: ' . $transaction_id);
        
        $idOperation = FrakmentaCommonUtils::clean_operation_id_frakmenta($transaction_id);
        Logger::log_info('Operation ID limpio: ' . $idOperation);
        Logger::log_info('Order ID: ' . $order->get_id());
        Logger::log_info('Estado actual de la orden: ' . $order->get_status());

        // Check if simulation mode is enabled
        $isTestMode = (get_option('FRAKMENTA_MODE') == 0); // 0 = test, 1 = production
        $simulationEnabled = defined('FRAKMENTA_CONFIRMATION_SIMULATE') && FRAKMENTA_CONFIRMATION_SIMULATE === true;
        
        if ($isTestMode && $simulationEnabled) {
            Logger::log_info('╔══════════════════════════════════════════════════════════════╗');
            Logger::log_info('║   MODO SIMULACIÓN ACTIVADO - SIMULANDO PAGO EXITOSO   ║');
            Logger::log_info('╚══════════════════════════════════════════════════════════════╝');
            Logger::log_info('Modo: TEST');
            Logger::log_info('FRAKMENTA_CONFIRMATION_SIMULATE: true');
            
            // Simulate successful payment response
            $simulated_response = (object)[
                'status' => 'OK',
                'messages' => ['La operación ha sido aceptada'],
                'data' => (object)[
                    'operation_id' => $idOperation,
                    'status' => 'ACCEPTED'
                ]
            ];
            
            Logger::log_info('Respuesta simulada:');
            Logger::log_info(json_encode($simulated_response, JSON_PRETTY_PRINT));
            
            if ($order->get_status() === 'pending') {
                Logger::log_info('Actualizando orden a processing (SIMULADO)');
                $order->set_status('processing', 'Orden pagada por Frakmenta (SIMULADO)');
                wc_reduce_stock_levels($order->get_id());
                $order->save();
                Logger::log_info('Orden actualizada correctamente (SIMULADO)');
            } else {
                Logger::log_info('Orden ya estaba en estado: ' . $order->get_status());
            }
            
            return 'success';
        }
        
        // Normal flow - Real API call
        Logger::log_info('Modo normal - Verificando con API de Frakmenta');

        $signature = $this->generateStatusSignature(
            $frakmenta_parameters['FRAKMENTA_MERCHANT_ID'],
            $idOperation,
            $frakmenta_parameters['FRAKMENTA_PRIVATE_KEY']
        );

        $data = json_encode([
            'merchant_id' => $frakmenta_parameters['FRAKMENTA_MERCHANT_ID'],
            'delegation' => 1,
            'type' => 'e-commerce',
            'operation_id' => $idOperation,
            'signature' => $signature
        ]);

        Logger::log_info('Petición a /operations/status:');
        Logger::log_info($data);

        $response = $this->connectApi->connection(
            $frakmenta_parameters['FRAKMENTA_URL'] . '/api/fk/v2/operations/status',
            $data,
            'POST'
        );

        Logger::log_info('Respuesta de /operations/status:');
        Logger::log_info($response);

        $response_frakmenta = json_decode($response);

        if (!isset($response_frakmenta->messages[0])) {
            Logger::log_error('No hay mensaje en la respuesta de Frakmenta');
            Logger::log_debug('Respuesta completa: ' . print_r($response_frakmenta, true));
            return 'error';
        }

        Logger::log_info('Mensaje de Frakmenta: ' . $response_frakmenta->messages[0]);

        if ($response_frakmenta->messages[0] === 'La operación ha sido aceptada') {
            Logger::log_info('Operación aceptada por Frakmenta');
            
            if ($order->get_status() === 'pending') {
                Logger::log_info('Actualizando orden a processing');
                $order->set_status('processing', 'Orden pagada por Frakmenta');
                // HPOS compatible way to reduce stock
                wc_reduce_stock_levels($order->get_id());
                $order->save();
                Logger::log_info('Orden actualizada correctamente');
            } else {
                Logger::log_info('Orden ya estaba en estado: ' . $order->get_status());
            }
            
            return 'success';
        }

        Logger::log_warning('Operación no aceptada. Estado: ' . $response_frakmenta->messages[0]);
        return 'error';
    }

    /**
     * Generate signature for status check
     *
     * @param string $merchant_id
     * @param string $operation_id
     * @param string $private_key
     * @return string
     */
    private function generateStatusSignature(
        string $merchant_id,
        string $operation_id,
        string $private_key
    ): string {
        return hash(
            'sha256',
            $merchant_id . '|' .
            '1|' .
            'e-commerce|' .
            $operation_id . '|' .
            $private_key,
            false
        );
    }
}
