<?php declare(strict_types=1);

/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Payments
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Frakmenta\WooCommerce\PaymentMethods;

use Frakmenta\WooCommerce\Utils\Logger;
use Frakmenta\WooCommerce\PaymentMethods\FrakmentaEcommerce\FrakmentaPayment;
use Frakmenta\WooCommerce\PaymentMethods\FrakmentaEcommerce\FrakmentaPaymentBlocksSupport;
use Frakmenta\WooCommerce\Utils\FrakmentaCommonUtils;
use Frakmenta\WooCommerce\Services\OrderService;
use WC_Order;
use WP_REST_Request;
use WP_REST_Response;

/**
 * The payment methods controller.
 *
 * Defines all the functionalities needed to register the Payment Methods actions and filters
 *
 * @since   4.0.0
 */
class PaymentMethodsController {

    /**
     * Register the stylesheets related with the payment methods
     *
     * @see https://developer.wordpress.org/reference/functions/wp_enqueue_style/
     *
     * @return void
     */
    public function enqueue_styles(): void {
        if ( is_checkout() || is_product()) {
            wp_enqueue_style( 'frakmenta-commons-css', FRAKMENTA_PLUGIN_URL . '/assets/commons/css/frakmenta.css', array(), FRAKMENTA_PLUGIN_VERSION, 'all' );
            wp_enqueue_style( 'frakmenta-commons-css', FRAKMENTA_PLUGIN_URL . '/assets/commons/css/frakmenta_additionals.css', array(), FRAKMENTA_PLUGIN_VERSION, 'all' );
            wp_enqueue_style( 'frakmenta-commons-css', FRAKMENTA_PLUGIN_URL . '/assets/commons/css/frakmenta_style.css', array(), FRAKMENTA_PLUGIN_VERSION, 'all' );
            wp_enqueue_style( 'frakmenta-remote-widget-css', get_option('FRAKMENTA_URL').'/css/widget-ecommerce.css', array(), FRAKMENTA_PLUGIN_VERSION);
        }
        
        if ( is_checkout() ) {
            wp_enqueue_style( 'frakmenta-modal-css', FRAKMENTA_PLUGIN_URL . '/assets/payments/css/frakmenta_modal.css', array(), FRAKMENTA_PLUGIN_VERSION, 'all' );
        }
    }

    /**
     * Get product price
     *
     * @return string
     */
    private function get_product_price(): string
    {
        global $product;

        if (!is_a($product, 'WC_Product')) {
            $product = wc_get_product(get_the_ID());
        }

        if (!$product) {
            return '0';
        }

        $price = wc_get_price_including_tax($product);

        return strval($price);
    }

    /**
     * Register and enqueue scripts for payment methods
     *
     * @return void
     */
    public function enqueue_scripts(): void {
        $frakmenta_params = array(
            'FRAKMENTA_URL' => get_option('FRAKMENTA_URL'),
            'FRAKMENTA_PUBLIC_KEY' => get_option('FRAKMENTA_PUBLIC_KEY'),
            'FRAKMENTA_LOGO' => FRAKMENTA_PLUGIN_URL . '/assets/commons/img/logo_frakmenta.png',
            'fkPlatform'           => 'woocommerce',
            'fkPlatformVersion'    => defined('WC_VERSION') ? WC_VERSION : ''
        );

        if (is_product() && !empty(get_option('FRAKMENTA_PRODUCT_OPTION'))) {
            $this->enqueueProductScripts($frakmenta_params);
        }

        if (is_checkout()) {
            $this->enqueueCheckoutScripts($frakmenta_params);
        }
    }

    /**
     * Enqueue product page scripts
     *
     * @param array $frakmenta_params
     * @return void
     */
    private function enqueueProductScripts(array $frakmenta_params): void {
        $frakmenta_params['FRAKMENTA_PRODUCT_PRICE'] = $this->get_product_price();

        // Add custom attributes to the widget script tag
        add_filter('script_loader_tag', function($tag, $handle, $src) use ($frakmenta_params) {
            if ($handle === 'frakmenta-product-remote-js') {
                $tag = str_replace(
                    ' src=',
                    ' data-apikey="' . esc_attr($frakmenta_params['FRAKMENTA_PUBLIC_KEY']) . '"' .
                    ' data-name="widgetFK"' .
                    ' data-api-url="' . esc_attr($frakmenta_params['FRAKMENTA_URL']) . '"' .
                    ' data-ecommerce-url="' . esc_attr($frakmenta_params['FRAKMENTA_URL']) . '"' .
                    ' src=',
                    $tag
                );
            }
            return $tag;
        }, 10, 3);

        wp_register_script(
            'frakmenta-product-local-js',
            FRAKMENTA_PLUGIN_URL . '/assets/products/js/frakmenta_front_products.js',
            array('jquery'),
            FRAKMENTA_PLUGIN_VERSION,
            true
        );
        wp_localize_script('frakmenta-product-local-js', 'frakmentaParams', $frakmenta_params);
        wp_enqueue_script('frakmenta-product-local-js');

        wp_register_script(
            'frakmenta-product-remote-js',
            get_option('FRAKMENTA_URL') . '/js/widgetEcommerce.js',
            array('jquery'),
            FRAKMENTA_PLUGIN_VERSION,
            true
        );
        wp_localize_script('frakmenta-product-remote-js', 'frakmentaParams', $frakmenta_params);
        $inline_js = '
        window.fkPlatform        = "woocommerce";
        window.fkPlatformVersion = ' . json_encode( defined('WC_VERSION') ? WC_VERSION : '' ) . ';';
        wp_add_inline_script('frakmenta-product-remote-js', $inline_js, 'before');
        wp_enqueue_script('frakmenta-product-remote-js');
    }

    /**
     * Enqueue checkout page scripts
     *
     * @param array $frakmenta_params
     * @return void
     */
    private function enqueueCheckoutScripts(array $frakmenta_params): void {
        // Check if we're using WooCommerce Blocks checkout
        $is_blocks_checkout = has_block('woocommerce/checkout');
        
        if ($is_blocks_checkout) {
            Logger::log_info('Checkout de Bloques detectado - scripts manejados por FrakmentaPaymentBlocksSupport');
            // Don't load classic checkout scripts, blocks will handle everything
            return;
        }
        
        Logger::log_info('📄 Checkout clásico detectado - cargando scripts clásicos');
        
        // Add custom attributes to the widget script tag
        add_filter('script_loader_tag', function($tag, $handle, $src) use ($frakmenta_params) {
            if ($handle === 'frakmenta-payment-remote-js') {
                $tag = str_replace(
                    ' src=',
                    ' data-apikey="' . esc_attr($frakmenta_params['FRAKMENTA_PUBLIC_KEY']) . '"' .
                    ' data-name="widgetFK"' .
                    ' data-api-url="' . esc_attr($frakmenta_params['FRAKMENTA_URL']) . '"' .
                    ' data-ecommerce-url="' . esc_attr($frakmenta_params['FRAKMENTA_URL']) . '"' .
                    ' src=',
                    $tag
                );
            }
            return $tag;
        }, 10, 3);
        
        wp_register_script(
            'frakmenta-payment-remote-js',
            get_option('FRAKMENTA_URL') . '/js/widgetEcommerce.js',
            array('jquery'),
            FRAKMENTA_PLUGIN_VERSION,
            true
        );
        wp_localize_script('frakmenta-payment-remote-js', 'frakmentaParams', $frakmenta_params);
        $inline_js = '
        window.fkPlatform        = "woocommerce";
        window.fkPlatformVersion = ' . json_encode( defined('WC_VERSION') ? WC_VERSION : '' ) . ';';
        wp_add_inline_script('frakmenta-payment-remote-js', $inline_js, 'before');
        wp_enqueue_script('frakmenta-payment-remote-js');

        wp_register_script(
            'frakmenta-payment-local-js',
            FRAKMENTA_PLUGIN_URL . '/assets/payments/js/frakmenta_check_out.js',
            array('jquery'),
            FRAKMENTA_PLUGIN_VERSION,
            true
        );
        wp_localize_script('frakmenta-payment-local-js', 'frakmentaParams', $frakmenta_params);
        wp_enqueue_script('frakmenta-payment-local-js');
        
        // Modal scripts
        wp_register_script(
            'frakmenta-modal-js',
            FRAKMENTA_PLUGIN_URL . '/assets/payments/js/frakmenta-modal.js',
            array('jquery'),
            FRAKMENTA_PLUGIN_VERSION,
            true
        );
        wp_localize_script('frakmenta-modal-js', 'frakmentaParams', $frakmenta_params);
        wp_enqueue_script('frakmenta-modal-js');
        
        wp_register_script(
            'frakmenta-checkout-handler-js',
            FRAKMENTA_PLUGIN_URL . '/assets/payments/js/frakmenta-checkout-handler.js',
            array('jquery', 'frakmenta-modal-js'),
            FRAKMENTA_PLUGIN_VERSION,
            true
        );
        wp_enqueue_script('frakmenta-checkout-handler-js');
    }

    /**
     * Merge existing gateways and Frakmenta
     *
     * @param array $gateways
     * @return array
     */
    public static function set_gateway( array $gateways ): array {
        Logger::log_info('Registrando Frakmenta como gateway de pago');
        Logger::log_debug('Gateways actuales: ' . print_r(array_keys($gateways), true));
        
        $gateways[] = FrakmentaPayment::class;
        
        Logger::log_info('Frakmenta registrado correctamente. Total gateways: ' . count($gateways));
        return $gateways;
    }

    /**
     * Filter the payment methods by the countries defined in their settings
     *
     * @param   array $payment_gateways
     * @return  array
     */
    public function filter_gateway_per_country( array $payment_gateways ): array {
        $customer_country = ( WC()->customer ) ? WC()->customer->get_billing_country() : false;
        
        Logger::log_debug('Filtrando gateways por país. País del cliente: ' . ($customer_country ?: 'No definido'));
        Logger::log_debug('Gateways disponibles antes de filtrar: ' . print_r(array_keys($payment_gateways), true));
        
        // Fix: Ensure gateways array uses gateway ID as key, not numeric indices
        $fixed_gateways = array();
        foreach ( $payment_gateways as $gateway_id => $gateway ) {
            if (is_object($gateway) && isset($gateway->id)) {
                // Use gateway's actual ID as array key
                $fixed_gateways[$gateway->id] = $gateway;
                Logger::log_debug('Gateway [' . $gateway_id . '] => Reindexado como [' . $gateway->id . '] ' . get_class($gateway));
            } else {
                // Keep as-is if not an object with ID
                $fixed_gateways[$gateway_id] = $gateway;
            }
        }
        
        $payment_gateways = $fixed_gateways;
        Logger::log_debug('Array después de reindexar: ' . print_r(array_keys($payment_gateways), true));
        
        // Now filter by country
        foreach ( $payment_gateways as $gateway_id => $gateway ) {
            if ( ! empty( $gateway->countries ) && $customer_country && ! in_array( $customer_country, $gateway->countries, true ) ) {
                Logger::log_debug('Gateway ' . $gateway_id . ' eliminado por restricción de país');
                unset( $payment_gateways[ $gateway_id ] );
            }
        }
        
        Logger::log_debug('Gateways disponibles después de filtrar: ' . print_r(array_keys($payment_gateways), true));
        Logger::log_debug('Total gateways finales: ' . count($payment_gateways));
        
        return $payment_gateways;
    }

    /**
     * Debug method to see final available gateways and fix array keys
     *
     * @param   array $payment_gateways
     * @return  array
     */
    public function debug_available_gateways( array $payment_gateways ): array {
        Logger::log_info('=== GATEWAYS FINALES EN CHECKOUT - CORRIGIENDO ÍNDICES ===');
        
        // Mostrar estructura completa ANTES
        $before_keys = '';
        foreach ($payment_gateways as $k => $v) {
            $before_keys .= '[' . $k . '] => ' . (is_object($v) ? $v->id : 'N/A') . ', ';
        }
        Logger::log_debug('Estructura ANTES: ' . $before_keys);
        
        // FIX CRÍTICO: Reindexar el array usando el ID del gateway como key
        $fixed_gateways = array();
        foreach ( $payment_gateways as $gateway_id => $gateway ) {
            if (is_object($gateway) && isset($gateway->id)) {
                // Usar el ID real del gateway como key
                $fixed_gateways[$gateway->id] = $gateway;
                Logger::log_debug('Guardando con key: "' . $gateway->id . '"');
            } else {
                Logger::log_warning('Gateway sin ID o no es objeto: [' . $gateway_id . ']');
                $fixed_gateways[$gateway_id] = $gateway;
            }
        }
        
        // Mostrar estructura completa DESPUÉS
        $after_keys = '';
        foreach ($fixed_gateways as $k => $v) {
            $after_keys .= '[' . $k . '] => ' . (is_object($v) ? $v->id : 'N/A') . ', ';
        }
        Logger::log_info('Estructura DESPUÉS: ' . $after_keys);
        
        if (!isset($fixed_gateways['frakmenta'])) {
            Logger::log_error('FRAKMENTA NO ESTÁ CON KEY "frakmenta"');
        } else {
            Logger::log_info('Frakmenta disponible con key "frakmenta" - ID: ' . $fixed_gateways['frakmenta']->id);
        }
        
        return $fixed_gateways;
    }

    /**
     * Check payment status on thank you page
     *
     * @param int|string $order_id The order number id received in callback thankyou page
     *
     * @return void
     */
    public function check_payment_frakmenta($order_id): void
    {
        Logger::log_info('╔══════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   CHECK_PAYMENT_FRAKMENTA EJECUTADO   ║');
        Logger::log_info('╚══════════════════════════════════════════════════════════════╝');
        Logger::log_info('Order ID recibido: ' . $order_id);
        
        if (empty($order_id)) {
            Logger::log_warning('Order ID está vacío');
            return;
        }

        $order = wc_get_order($order_id);
        
        if (!$order) {
            Logger::log_error('No se pudo obtener la orden ' . $order_id);
            return;
        }

        Logger::log_info('Método de pago de la orden: ' . $order->get_payment_method());

        if ($order->get_payment_method() !== 'frakmenta') {
            Logger::log_info('⏩ Orden no es de Frakmenta, saltando verificación');
            return;
        }

        Logger::log_info('Es una orden de Frakmenta, verificando estado...');

        $orderService = new OrderService();
        $result = $orderService->getStatusOperationFrakmenta($order);
        
        Logger::log_info('Resultado de verificación: ' . $result);
        
        if ($result === 'success') {
            Logger::log_info('Pago verificado exitosamente');
            $transactionFrakmenta = FrakmentaCommonUtils::clean_operation_id_frakmenta(
                $order->get_transaction_id()
            );
            Logger::log_info('Cargando template con transaction ID: ' . $transactionFrakmenta);
            require_once FRAKMENTA_PLUGIN_DIR_PATH . 'templates/fk_transaction_details.php';
        } else {
            Logger::log_warning('Verificación de pago falló');
        }
    }

    /**
     * Register Frakmenta payment method for WooCommerce Blocks
     *
     * @param \Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry
     * @return void
     */
    public function register_blocks_support($payment_method_registry): void {
        Logger::log_info('╔══════════════════════════════════════════════════════════╗');
        Logger::log_info('║   REGISTRANDO FRAKMENTA BLOCKS SUPPORT   ║');
        Logger::log_info('╚══════════════════════════════════════════════════════════╝');
        
        $payment_method_registry->register(new FrakmentaPaymentBlocksSupport());
        
        Logger::log_info('FrakmentaPaymentBlocksSupport registrado en el registry');
    }
}
