<?php declare(strict_types=1);

namespace Frakmenta\WooCommerce\PaymentMethods\FrakmentaEcommerce;

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;
use Frakmenta\WooCommerce\Utils\Logger;

/**
 * Frakmenta Blocks integration
 */
final class FrakmentaPaymentBlocksSupport extends AbstractPaymentMethodType {

    /**
     * Payment method name/id/slug.
     *
     * @var string
     */
    protected $name = 'frakmenta';

    /**
     * The gateway instance.
     *
     * @var FrakmentaPayment
     */
    private $gateway;

    /**
     * Initializes the payment method type.
     */
    public function initialize() {
        Logger::log_info('╔════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   FRAKMENTA BLOCKS SUPPORT INICIALIZANDO   ║');
        Logger::log_info('╚════════════════════════════════════════════════════════════╝');
        
        $this->settings = get_option('woocommerce_frakmenta_settings', []);
        $gateways       = WC()->payment_gateways->payment_gateways();
        $this->gateway  = $gateways['frakmenta'] ?? new FrakmentaPayment();
        
        // Add filter to inject custom attributes into the script tag
        add_filter('script_loader_tag', [$this, 'add_script_attributes'], 10, 2);
        
        Logger::log_info('Blocks support inicializado para Frakmenta');
    }
    
    /**
     * Add custom attributes to the remote widget script tag
     */
    public function add_script_attributes($tag, $handle) {
        if ($handle === 'frakmenta-widget-remote-blocks') {
            $frakmenta_public_key = get_option('FRAKMENTA_PUBLIC_KEY');
            $frakmenta_url = get_option('FRAKMENTA_URL');
            Logger::log_info('Inyectando atributos data-apikey, data-name, data-api-url y data-ecommerce-url en script remoto');
            $tag = str_replace(
                ' src=',
                ' data-apikey="' . esc_attr($frakmenta_public_key) . '"' .
                ' data-name="widgetFK"' .
                ' data-api-url="' . esc_attr($frakmenta_url) . '"' .
                ' data-ecommerce-url="' . esc_attr($frakmenta_url) . '"' .
                ' src=',
                $tag
            );
        }
        return $tag;
    }

    /**
     * Returns if this payment method should be active.
     *
     * @return boolean
     */
    public function is_active() {
        $is_active = $this->gateway->is_available();
        Logger::log_info('Blocks: is_active() = ' . ($is_active ? 'TRUE' : 'FALSE'));
        return $is_active;
    }

    /**
     * Returns an array of scripts/handles to be registered for this payment method.
     *
     * @return array
     */
    public function get_payment_method_script_handles() {
        static $scripts_registered = false;
        
        Logger::log_info('Blocks: get_payment_method_script_handles llamado');
        
        if ($scripts_registered) {
            Logger::log_info('Scripts ya registrados en esta petición, retornando handle');
            return ['wc-frakmenta-blocks'];
        }
        
        Logger::log_info('Blocks: Registrando scripts para Frakmenta');
        
        // Register the remote Frakmenta widget script
        $frakmenta_url = get_option('FRAKMENTA_URL');
        
        if (!wp_script_is('frakmenta-widget-remote-blocks', 'registered')) {
            wp_register_script(
                'frakmenta-widget-remote-blocks',
                $frakmenta_url . '/js/widgetEcommerce.js',
                [],
                FRAKMENTA_PLUGIN_VERSION,
                true
            );
            Logger::log_info('Script frakmenta-widget-remote-blocks registrado');
        }
        
        // Register modal script and style
        if (!wp_script_is('frakmenta-modal-blocks', 'registered')) {
            wp_register_script(
                'frakmenta-modal-blocks',
                FRAKMENTA_PLUGIN_URL . '/assets/payments/js/frakmenta-modal.js',
                ['jquery'],
                FRAKMENTA_PLUGIN_VERSION,
                true
            );
            Logger::log_info('Script frakmenta-modal-blocks registrado');
        }
        
        if (!wp_style_is('frakmenta-modal-blocks', 'registered')) {
            wp_register_style(
                'frakmenta-modal-blocks',
                FRAKMENTA_PLUGIN_URL . '/assets/payments/css/frakmenta_modal.css',
                [],
                FRAKMENTA_PLUGIN_VERSION
            );
            wp_enqueue_style('frakmenta-modal-blocks');
            Logger::log_info('Style frakmenta-modal-blocks registrado y encolado');
        }
        
        $script_path       = '/assets/payments/js/frakmenta-blocks.js';
        $script_asset_path = FRAKMENTA_PLUGIN_DIR_PATH . 'assets/payments/js/frakmenta-blocks.asset.php';
        $script_asset      = file_exists($script_asset_path)
            ? require $script_asset_path
            : [
                'dependencies' => ['wc-blocks-registry', 'wc-settings', 'wp-element', 'wp-html-entities', 'wp-i18n'],
                'version'      => FRAKMENTA_PLUGIN_VERSION
            ];
        $script_url        = FRAKMENTA_PLUGIN_URL . $script_path;

        if (!wp_script_is('wc-frakmenta-blocks', 'registered')) {
            wp_register_script(
                'wc-frakmenta-blocks',
                $script_url,
                array_merge($script_asset['dependencies'], ['frakmenta-widget-remote-blocks', 'frakmenta-modal-blocks']),
                FRAKMENTA_PLUGIN_VERSION . '-' . time(), // Force cache bust
                true
            );

            if (function_exists('wp_set_script_translations')) {
                wp_set_script_translations('wc-frakmenta-blocks', 'frakmenta', FRAKMENTA_PLUGIN_DIR_PATH . 'languages');
            }
            
            Logger::log_info('Script wc-frakmenta-blocks registrado');
        }
        
        $scripts_registered = true;

        return ['wc-frakmenta-blocks'];
    }

    /**
     * Returns an array of key=>value pairs of data made available to the payment methods script.
     *
     * @return array
     */
    public function get_payment_method_data() {
        Logger::log_info('Blocks: Proporcionando datos para Frakmenta');
        
        $limits = $this->gateway->get_payment_mechant_limits();
        
        $cart_total = WC()->cart ? floatval(WC()->cart->total) : 0;
        
        // Get dynamic title based on cart total
        $title = $this->gateway->get_payment_method_title_checkout(
            $limits['min_import'] ?? 59,
            $limits['max_import'] ?? 1000
        );
        
        Logger::log_debug('Título calculado para bloques: ' . $title);
        
        $data = [
            'title'       => $title,
            'description' => $this->gateway->description,
            'supports'    => array_filter($this->gateway->supports, [$this->gateway, 'supports']),
            //'logo_url'    => FRAKMENTA_PLUGIN_URL . '/assets/commons/img/logo.png', //2025-12-11: Logo removed from blocks
            'min_amount'  => $limits['min_import'] ?? 59,
            'max_amount'  => $limits['max_import'] ?? 1000,
            'cart_total'  => $cart_total,
            'frakmenta_url' => get_option('FRAKMENTA_URL'),
            'public_key'  => get_option('FRAKMENTA_PUBLIC_KEY'),
            'has_fields'  => $this->gateway->has_fields(),
        ];
        
        Logger::log_debug('Datos enviados a blocks: ' . print_r($data, true));
        Logger::log_info('Datos preparados para el frontend de bloques');
        
        return $data;
    }
}
