<?php declare(strict_types=1);

/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Connect
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Frakmenta\WooCommerce\PaymentMethods\Base;

/**
 * Payment Method Interface
 *
 * Defines the contract that all Frakmenta payment methods must implement.
 * This interface ensures consistency across different payment method implementations.
 *
 * @package Frakmenta\WooCommerce\PaymentMethods\Base
 * @since   1.0.0
 */
interface PaymentMethodInterface {

    /**
     * Get the unique identifier of the payment method
     *
     * Returns the payment gateway ID used by WooCommerce to identify this payment method.
     * Must be unique across all payment gateways.
     *
     * @since  1.0.0
     * @return string Payment method ID (e.g., 'frakmenta')
     */
    public function get_payment_method_id(): string;

    /**
     * Get the internal code of the payment method
     *
     * Returns the internal payment method code used in API communications.
     *
     * @since  1.0.0
     * @return string Payment method code (e.g., 'FRAKMENTA')
     */
    public function get_payment_method_code(): string;

    /**
     * Get the transaction type of the payment method
     *
     * Indicates whether the payment method uses direct or redirect flow.
     * - 'direct': Customer completes payment on WooCommerce checkout
     * - 'redirect': Customer is redirected to external payment page
     *
     * @since  1.0.0
     * @return string Transaction type: 'direct' or 'redirect'
     */
    public function get_payment_method_type(): string;

    /**
     * Get the title displayed in WooCommerce backend
     *
     * Returns the payment method name shown in admin panel.
     *
     * @since  1.0.0
     * @return string Payment method title for admin area
     */
    public function get_payment_method_title(): string;

    /**
     * Get the description displayed in WooCommerce backend
     *
     * Returns the payment method description shown in admin panel
     * payment settings page.
     *
     * @since  1.0.0
     * @return string Payment method description for admin area
     */
    public function get_payment_method_description(): string;

    /**
     * Check if payment method has custom checkout fields
     *
     * Indicates whether this payment method requires additional
     * custom fields in the checkout page.
     *
     * @since  1.0.0
     * @return bool True if has custom fields, false otherwise
     */
    public function has_fields(): bool;

    /**
     * Get array of custom checkout field IDs
     *
     * Returns the IDs of custom fields that should be displayed
     * in the checkout page when this payment method is selected.
     *
     * @since  1.0.0
     * @return array<int, string> Array of field IDs (e.g., ['salutation', 'birthday'])
     */
    public function get_checkout_fields_ids(): array;

    /**
     * Get the filename of the payment method icon
     *
     * Returns the icon filename to be displayed next to the payment method
     * in checkout. File should exist in assets/commons/img/ directory.
     *
     * @since  1.0.0
     * @return string Icon filename (e.g., 'logo.png')
     */
    public function get_payment_method_icon(): string;

    /**
     * Get merchant operation limits from Frakmenta
     *
     * Retrieves the minimum and maximum transaction amounts allowed
     * for this merchant from Frakmenta API.
     *
     * @since  1.0.0
     * @return array{max_import: float|int, min_import: float|int} Array with:
     *               - max_import: Maximum transaction amount
     *               - min_import: Minimum transaction amount
     */
    public function get_payment_mechant_limits(): array;
}
