<?php declare( strict_types=1 );
/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Connect
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
namespace Frakmenta\WooCommerce\PaymentMethods\Base;

use Exception;
use Frakmenta\WooCommerce\Services\OrderService;
use Frakmenta\WooCommerce\Utils\Logger;
use WC_Countries;
use WC_Payment_Gateway;
use WP_Error;
use WC_Gateway_COD;

abstract class BasePaymentMethod extends WC_Payment_Gateway implements PaymentMethodInterface {

    /**
     * What type of transaction, should be 'direct' or 'redirect'
     *
     * @var string
     */
    protected $type;

    /**
     * An array with the keys of the required custom fields
     *
     * @var array
     */
    protected $checkout_fields_ids;

    /**
     * The minimun amount for the payment method
     *
     * @var string
     */
    public $min_amount;

    /**
     * A custom initialized order status for this payment method
     *
     * @var string
     */
    public $initial_order_status;

    /**
     * Frakmenta limits merchant
     *
     * @var string
     */
    public $frakmentaMerchantLimits;


    /**
     * Construct for Core class.
     */
    public function __construct() {

        Logger::log_info('=== INICIALIZANDO GATEWAY FRAKMENTA ===');

        $this->frakmentaMerchantLimits = $this->get_payment_mechant_limits();
        $this->max_amount          = $this->frakmentaMerchantLimits['max_import'];
        $this->min_amount          = $this->frakmentaMerchantLimits['min_import'];

        Logger::log_debug('Límites obtenidos - Min: ' . $this->min_amount . ', Max: ' . $this->max_amount);

        $this->id                  = $this->get_payment_method_id();
        //$this->icon                = $this->get_logo();
        $this->has_fields          = $this->has_fields();
        $this->method_title        = $this->get_payment_method_title();
        $this->method_description  = $this->get_payment_method_description();
        $this->supports            = array('products');

        $this->init_form_fields();
        $this->init_settings();
        
        $this->title                = $this->get_payment_method_title();
        $this->description          = $this->get_option('description');
        $this->enabled              = $this->get_option('enabled', 'no');
        $this->type                 = $this->get_payment_method_type();
        $this->countries            = $this->get_option('countries');
        $this->initial_order_status = $this->get_option('initial_order_status', false);
        $this->errors               = array();
        $this->checkout_fields_ids  = $this->get_checkout_fields_ids();

        Logger::log_info('Gateway Frakmenta inicializado:');
        Logger::log_debug('- ID: ' . $this->id);
        Logger::log_debug('- Title: ' . $this->title);
        Logger::log_debug('- Enabled: ' . $this->enabled);
        Logger::log_debug('- Has fields: ' . ($this->has_fields ? 'Yes' : 'No'));
        Logger::log_debug('- Type: ' . $this->type);
        Logger::log_debug('- Countries: ' . (is_array($this->countries) ? implode(', ', $this->countries) : 'All'));

        if (is_checkout()) {
            Logger::log_debug('Ejecutando updateCheckoutTitle (página de checkout)');
            $this->updateCheckoutTitle();
        }

        add_action(
            'woocommerce_update_options_payment_gateways_' . $this->id, 
            array($this, 'process_admin_options')
        );

        add_action(
            'woocommerce_update_options_payment_gateways_' . $this->id, 
            array($this, 'display_errors')
        );
        
        Logger::log_info('Construcción de Gateway Frakmenta completada');
    }

    /**
     * Update title for checkout page
     *
     * @return void
     */
    private function updateCheckoutTitle(): void {
        $this->method_title = $this->get_payment_method_title_checkout(
            floatval($this->frakmentaMerchantLimits['min_import']),
            floatval($this->frakmentaMerchantLimits['max_import'])
        );
        $this->title = $this->method_title;
    }

    /**
     * Return the full path of the (locale) logo
     *
     * @return string
     */
    private function get_logo(): string {
        $language = substr( get_locale(), 0, 2 );

        $icon = $this->get_payment_method_icon();

        $icon_locale = substr_replace( $icon, "-$language", - 4, - 4 );
        if ( file_exists( FRAKMENTA_PLUGIN_URL . 'assets/commons/img/' . $icon_locale ) ) {
            $icon = $icon_locale;
        }

        return esc_url( FRAKMENTA_PLUGIN_URL . '/assets/commons/img/' . $icon );
    }

    /**
     * Return an array of allowed countries defined in WooCommerce Settings.
     *
     * @return array
     */
    private function get_countries(): array {
        $countries = new WC_Countries();
        return $countries->get_allowed_countries();
    }

    /**
     * Return if payment methods requires custom checkout fields
     *
     * @return boolean
     */
    public function has_fields(): bool {
        return false;
    }

    /**
     * Return the custom checkout fields id`s
     *
     * @return array
     */
    public function get_checkout_fields_ids(): array {
        return array();
    }

    /**
     * Define the form option - settings fields.
     *
     * @return  array
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled'              => array(
                'title'   =>  'Enable/Disable',
                'label'   => 'Enable ' . $this->get_method_title() . ' Gateway',
                'type'    => 'checkbox',
                'default' => 'no',
            ),
            'title'                => array(
                'title'    => __( 'Title', 'Frakmenta' ),
                'type'     => 'text',
                'desc_tip' => __( 'This controls the title which the user sees during checkout.', 'frakmenta' ),
                'default'  => $this->get_method_title(),
            ),
            'initial_order_status' => array(
                'title'    => __( 'Initial Order Status', 'Frakmenta' ),
                'type'     => 'select',
                'options'  => $this->get_order_statuses(),
                'desc_tip' => __( 'Initial order status for this payment method.', 'Frakmenta' ),
                'default'  => 'wc-default',
            ),
            'countries'            => array(
                'title'       => __( 'Pais', 'Frakmenta' ),
                'type'        => 'multiselect',
                'description' => __( 'If you select one or more countries, this payment method will be shown in the checkout page, if the payment address`s country of the customer match with the selected values. Leave blank for no restrictions.', 'frakmenta' ),
                'desc_tip'    => __( 'For most operating system and configurations, you must hold Ctrl or Cmd in your keyboard, while you click in the options to select more than one value.', 'frakmenta' ),
                'options'     => $this->get_countries(),
                'default'     => $this->get_option( 'countries', array() ),
            ),
            'direct'  => array(
                'title' => __('Transaction Type', 'frakmenta'),
                /* translators: %1$: The payment method title */
                'label' => sprintf(__('Enable direct %1$s', 'frakmenta'), 'Frakmenta'),
                'type' => 'checkbox',
                'default' => 'yes',
                'desc_tip' => __('If enabled, additional information can be entered during WooCommerce checkout. If disabled, additional information will be requested on the Frakmenta payment page.', 'Frakmenta'),
            ),
        );
    }

    /**
     * Process the payment and return the result.
     *
     * @param integer $order_id Order ID.
     *
     * @return  array
     */
    public function process_payment($order_id): array {
        Logger::log_info('=== PROCESANDO PAGO FRAKMENTA ===');
        Logger::log_info('Order ID: ' . $order_id);
        
        try {
            $order_service = new OrderService();
            $order = wc_get_order($order_id);
            
            if (!$order) {
                Logger::log_error('Error: No se pudo obtener la orden ' . $order_id);
                throw new \Exception('No se pudo obtener la orden');
            }
            
            $order_total = floatval($order->get_total());
            Logger::log_info('Orden obtenida correctamente. Total: ' . $order_total);
            Logger::log_debug('Datos de la orden: ' . print_r([
                'id' => $order->get_id(),
                'total' => $order_total,
                'currency' => $order->get_currency(),
                'customer_email' => $order->get_billing_email()
            ], true));
            
            // Validar límites de importe
            $min_amount = floatval($this->min_amount);
            $max_amount = floatval($this->max_amount);
            
            Logger::log_info('Validando límites - Min: ' . $min_amount . '€, Max: ' . $max_amount . '€');
            
            if ($order_total < $min_amount) {
                $remaining = $min_amount - $order_total;
                $error_msg = sprintf('El importe del pedido (%.2f€) es menor al mínimo requerido (%.2f€), para pagar con Frakmenta.', $order_total, $min_amount);
                Logger::log_error($error_msg);
                wc_add_notice($error_msg, 'error');
                return array(
                    'result' => 'failure',
                    'messages' => $error_msg
                );
            }
            
            if ($order_total > $max_amount) {
                $error_msg = sprintf('El importe del pedido (%.2f€) supera el máximo permitido (%.2f€), para pagar con Frakmenta.', $order_total, $max_amount);
                Logger::log_error($error_msg);
                wc_add_notice($error_msg, 'error');
                return array(
                    'result' => 'failure',
                    'messages' => $error_msg
                );
            }
            
            Logger::log_info('Importe dentro de límites - procediendo con la creación de orden');
            
            $paymentUrl = $order_service->create_order_frakmenta($order);
            
            Logger::log_info('URL de pago generada: ' . $paymentUrl);
            Logger::log_info('Proceso de pago completado exitosamente');
            
            return array(
                'result'   => 'success',
                'redirect' => esc_url_raw($paymentUrl),
                'frakmenta_modal' => true, // Flag to trigger modal
            );
        } catch (\Exception $e) {
            Logger::log_error('Error al procesar pago: ' . $e->getMessage());
            Logger::log_error('Stack trace: ' . $e->getTraceAsString());
            
            wc_add_notice('Error al procesar el pago con Frakmenta: ' . $e->getMessage(), 'error');
            
            return array(
                'result'   => 'failure',
                'messages' => 'Error al procesar el pago'
            );
        }
    }

    /**
     * Prints checkout custom fields
     *
     * @return  mixed
     */
    public function payment_fields() {
        Logger::log_info('╔═════════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   PAYMENT_FIELDS LLAMADO - FRAKMENTA INTENTA RENDERIZAR   ║');
        Logger::log_info('╚═════════════════════════════════════════════════════════════════╝');
        Logger::log_info('Gateway ID: ' . $this->id);
        Logger::log_info('Gateway Title: ' . $this->title);
        Logger::log_info('Gateway Enabled: ' . $this->enabled);
        
        global $woocommerce;
        $realTotalCart = $woocommerce->cart->total;
        $totalCart = $realTotalCart * 100;

        Logger::log_debug('Total del carrito para widget: ' . $realTotalCart . ' (en centavos: ' . $totalCart . ')');
        Logger::log_debug('Límites - Min: ' . $this->min_amount . ', Max: ' . $this->max_amount);

        if ($realTotalCart >= $this->min_amount && $realTotalCart <= $this->max_amount) {
            Logger::log_debug('Carrito dentro de límites - Mostrando campo de validación con value="1"');
            echo "<input type='hidden' name='validate_payfrakmenta' value='1' required readonly>";
        } else {
            Logger::log_warning('Carrito fuera de límites - Mostrando campo sin value');
            echo "<input type='hidden' name='validate_payfrakmenta' required readonly>";
        }

        Logger::log_debug('Renderizando widget de cuotas Frakmenta');
        echo "<div class='col-img-payment' style='margin-bottom:1em;'>";
        echo "<div class='fk-installments' id='fk-widget-installments' data-product_price='$totalCart'></div>";
        echo "</div>";
        
        Logger::log_info('╔═══════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   CAMPOS DE PAGO FRAKMENTA RENDERIZADOS COMPLETAMENTE   ║');
        Logger::log_info('╚═══════════════════════════════════════════════════════════════╝');
    }

    /**
     * Check if the gateway is available for use
     *
     * @return bool
     */
    public function is_available(): bool {
        Logger::log_info('╔════════════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   IS_AVAILABLE LLAMADO - VERIFICANDO DISPONIBILIDAD FRAKMENTA   ║');
        Logger::log_info('╚════════════════════════════════════════════════════════════════════╝');
        Logger::log_info('Gateway ID que está siendo verificado: ' . $this->id);
        Logger::log_info('Has fields: ' . ($this->has_fields() ? 'SÍ' : 'NO'));
        
        // First check if parent allows (checks enabled status)
        $parent_available = parent::is_available();
        Logger::log_info('Estado habilitado (parent::is_available): ' . ($parent_available ? 'SI' : 'NO'));
        Logger::log_debug('Enabled option value: ' . $this->enabled);
        
        if ( ! $parent_available ) {
            Logger::log_warning('Frakmenta NO disponible: Gateway deshabilitado en configuración');
            return false;
        }

        // Check if Frakmenta is configured
        $merchant_id = get_option('FRAKMENTA_MERCHANT_ID');
        $public_key = get_option('FRAKMENTA_PUBLIC_KEY');
        $frakmenta_url = get_option('FRAKMENTA_URL');
        $exist_account = get_option('FRAKMENTA_EXIST_ACCOUNT');
        
        Logger::log_debug('Configuración Frakmenta:');
        Logger::log_debug('- FRAKMENTA_MERCHANT_ID: ' . ($merchant_id ?: 'VACÍO'));
        Logger::log_debug('- FRAKMENTA_PUBLIC_KEY: ' . ($public_key ? substr($public_key, 0, 20) . '...' : 'VACÍO'));
        Logger::log_debug('- FRAKMENTA_URL: ' . ($frakmenta_url ?: 'VACÍO'));
        Logger::log_debug('- FRAKMENTA_EXIST_ACCOUNT: ' . ($exist_account ?: '0'));
        
        if ( empty($merchant_id) || empty($public_key) ) {
            Logger::log_warning('Frakmenta NO disponible: Faltan credenciales (merchant_id o public_key)');
            return false;
        }

        // Check cart total against limits
        if ( WC()->cart ) {
            $cart_total = floatval( WC()->cart->total );
            
            Logger::log_debug('Total del carrito: ' . $cart_total);
            Logger::log_debug('Límites Frakmenta - Min: ' . $this->min_amount . ', Max: ' . $this->max_amount);
            
            if ( $cart_total <= 0 ) {
                Logger::log_warning('Frakmenta NO disponible: Carrito vacío o total <= 0');
                return false;
            }

            // If cart total is outside limits, still show but with informative message
            // The title will be updated by updateCheckoutTitle() method
        }

        Logger::log_info('╔═══════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   FRAKMENTA DISPONIBLE - TODAS LAS VALIDACIONES OK   ║');
        Logger::log_info('╚═══════════════════════════════════════════════════════════════╝');
        return true;
    }

    /**
     * Validate_fields
     *
     * @return  boolean
     */
    public function validate_fields(): bool {
        Logger::log_info('╔════════════════════════════════════════════════════════════╗');
        Logger::log_info('║   VALIDATE_FIELDS LLAMADO   ║');
        Logger::log_info('╚════════════════════════════════════════════════════════════╝');
        
        // Check if it's a REST API request (WooCommerce Blocks)
        $is_rest_api = defined('REST_REQUEST') && REST_REQUEST;
        
        Logger::log_info('¿Es petición REST API (Blocks)?: ' . ($is_rest_api ? 'SÍ' : 'NO'));
        
        // For WooCommerce Blocks, validation is different
        if ($is_rest_api) {
            Logger::log_info('Validación omitida para WooCommerce Blocks (se maneja en el store)');
            return true;
        }
        
        // Traditional checkout validation
        $validate_field = isset($_POST['validate_payfrakmenta']) 
            ? sanitize_text_field(wp_unslash($_POST['validate_payfrakmenta'])) 
            : '';
        
        Logger::log_debug('Campo validate_payfrakmenta: ' . ($validate_field !== '' ? 'PRESENTE' : 'AUSENTE'));
            
        if ('' === $validate_field) {
            Logger::log_warning('Campo validate_payfrakmenta vacío - agregando error');
            wc_add_notice(
                __('Lo sentimos, tu transacción no cumple con los requisitos para ser financiada por Frakmenta', 'frakmenta'),
                'error'
            );
        }

        if (wc_get_notices('error')) {
            Logger::log_error('Validación fallida - hay errores de WooCommerce');
            return false;
        }

        Logger::log_info('Validación exitosa');
        return true;
    }

    /**
     * Returns the WooCommerce registered order statuses
     *
     * @see     http://hookr.io/functions/wc_get_order_statuses/
     *
     * @return  array
     */
    private function get_order_statuses(): array {
        $order_statuses               = wc_get_order_statuses();
        $order_statuses['wc-default'] = __( 'Default value set in common settings', 'frakmenta' );
        return $order_statuses;
    }
}
