<?php declare(strict_types=1);

/**
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade the Frakmenta plugin
 * to newer versions in the future. If you wish to customize the plugin for your
 * needs please document your changes and make backups before you update.
 *
 * @category    Frakmenta
 * @package     Connect
 * @author      Sistemas Findirect <desarrollo-frakmenta@findirect.com>
 * @copyright   Copyright (c) Frakmenta, Findirect. (https://www.frakmenta.com)
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Frakmenta\WooCommerce\Exceptions;

/**
 * Missing Dependency Exception
 *
 * Custom exception thrown when required plugin dependencies are not active.
 * Used during plugin activation to ensure all requirements are met.
 *
 * @package Frakmenta\WooCommerce\Exceptions
 * @since   4.0.0
 */
class MissingDependencyException extends \Exception {

    /**
     * The missing required plugins names
     *
     * @since 4.0.0
     * @var   array<int, string>
     */
    private array $missing_plugin_names;

    /**
     * MissingDependencyException constructor
     *
     * Creates a new exception with the list of missing plugin names.
     *
     * @since 4.0.0
     * @param array<int, string> $missing_plugins Array of missing plugin names (e.g., ['WooCommerce'])
     */
    public function __construct(array $missing_plugins) {
        parent::__construct();
        $this->missing_plugin_names = $missing_plugins;
    }

    /**
     * Get the list of all missing plugin names
     *
     * Returns the array of plugin names that are required but not currently active.
     *
     * @since  4.0.0
     * @return array<int, string> Array of missing plugin names
     *
     * @example
     * try {
     *     $checker = new DependencyChecker();
     *     $checker->check();
     * } catch (MissingDependencyException $e) {
     *     $missing = $e->get_missing_plugin_names();
     *     // Returns: ['WooCommerce']
     * }
     */
    public function get_missing_plugin_names(): array {
        return $this->missing_plugin_names;
    }

}
