/**
 * Frakmenta Payment Modal Handler
 */
(function($) {
    'use strict';

    console.log('Frakmenta Modal Script cargado');

    const FrakmentaModal = {
        modal: null,
        iframe: null,
        loading: null,
        paymentUrl: null,

        init: function() {
            console.log('Inicializando Frakmenta Modal');
            this.createModal();
            this.bindEvents();
        },

        createModal: function() {
            console.log('Creando estructura del modal');
            
            const modalHTML = `
                <div id="frakmenta-payment-modal">
                    <div class="frakmenta-modal-content">
                        <div class="frakmenta-modal-header">
                            <h3>Proceso de Pago - Frakmenta</h3>
                            <button type="button" class="frakmenta-modal-close" aria-label="Cerrar">&times;</button>
                        </div>
                        <div class="frakmenta-modal-body">
                            <div class="frakmenta-modal-loading">
                                <div class="frakmenta-spinner"></div>
                                <p>Cargando proceso de pago...</p>
                            </div>
                            <iframe id="frakmenta-payment-iframe" title="Frakmenta Payment"></iframe>
                        </div>
                    </div>
                </div>
            `;

            // Remove existing modal if any
            $('#frakmenta-payment-modal').remove();
            
            // Append to body
            $('body').append(modalHTML);

            this.modal = $('#frakmenta-payment-modal');
            this.iframe = $('#frakmenta-payment-iframe');
            this.loading = $('.frakmenta-modal-loading');

            console.log('Modal creado correctamente');
        },

        bindEvents: function() {
            console.log('Vinculando eventos del modal');
            
            // Close button
            $(document).on('click', '.frakmenta-modal-close', (e) => {
                e.preventDefault();
                console.log('Usuario cerró el modal');
                this.close();
            });

            // Close on overlay click
            this.modal.on('click', (e) => {
                if (e.target.id === 'frakmenta-payment-modal') {
                    console.log('Click en overlay - cerrando modal');
                    this.close();
                }
            });

            // Close on ESC key
            $(document).on('keydown', (e) => {
                if (e.key === 'Escape' && this.modal.hasClass('show')) {
                    console.log('ESC presionado - cerrando modal');
                    this.close();
                }
            });

            // Iframe load event
            this.iframe.on('load', () => {
                console.log('Iframe cargado');
                this.loading.addClass('hidden');
            });

            // Listen for messages from iframe (payment completion)
            window.addEventListener('message', (event) => {
                this.handleMessage(event);
            });

            console.log('Eventos vinculados correctamente');
        },

        open: function(url) {
            console.log('Abriendo modal con URL:', url);
            
            if (!url) {
                console.error('Error: URL no proporcionada');
                return;
            }

            this.paymentUrl = url;
            this.loading.removeClass('hidden');
            this.iframe.attr('src', url);
            this.modal.addClass('show');
            $('body').css('overflow', 'hidden');

            console.log('Modal abierto correctamente');
        },

        close: function() {
            console.log('🔒 Cerrando modal');
            
            this.modal.removeClass('show');
            $('body').css('overflow', '');
            
            // Clear iframe after animation
            setTimeout(() => {
                this.iframe.attr('src', 'about:blank');
                this.loading.removeClass('hidden');
            }, 300);

            console.log('Modal cerrado');
            
            // Reload page to reset checkout state
            console.log('Recargando página para resetear estado del checkout');
            setTimeout(() => {
                window.location.reload();
            }, 500);
        },

        handleMessage: function(event) {
            console.log('Mensaje recibido del iframe:', event.data);

            // Frakmenta puede enviar mensajes con diferentes formatos
            // Intentamos detectar la finalización del flujo de varias formas
            
            // Caso 1: Mensaje estructurado de Frakmenta
            if (event.data && typeof event.data === 'object') {
                if (event.data.type === 'frakmenta_payment_complete' || event.data.status === 'success') {
                    console.log('Pago completado exitosamente - redirigiendo a página de confirmación');
                    this.close(false); // Close without reload
                    
                    // Redirect to thank you page
                    if (event.data.redirect_url) {
                        window.location.href = event.data.redirect_url;
                    } else {
                        // Try to get success URL from order
                        window.location.href = window.location.origin + '/?wc-ajax=update_order_review';
                    }
                    return;
                }

                if (event.data.type === 'frakmenta_payment_cancelled' || event.data.status === 'cancelled') {
                    console.log('Pago cancelado - recargando checkout');
                    this.close(); // Close with reload
                    return;
                }

                if (event.data.type === 'frakmenta_payment_error' || event.data.status === 'error') {
                    console.error('Error en el pago:', event.data.message);
                    alert('Error al procesar el pago: ' + (event.data.message || 'Error desconocido'));
                    this.close(); // Close with reload
                    return;
                }
            }
            
            // Caso 2: Frakmenta redirige internamente al success_url o ko_url
            // Detectamos cambios de URL en el iframe
            try {
                const iframeUrl = this.iframe[0].contentWindow.location.href;
                console.log('URL actual del iframe:', iframeUrl);
                
                // Si el iframe fue redirigido a una URL de WooCommerce (success o checkout)
                if (iframeUrl.includes(window.location.origin)) {
                    console.log('Frakmenta redirigió a URL local - extrayendo y redirigiendo');
                    this.close(false); // Close without reload
                    window.location.href = iframeUrl;
                }
            } catch (e) {
                // CORS prevents reading iframe URL from different origin
                // This is expected and normal
            }
        },

        close: function(shouldReload = true) {
            console.log('🔒 Cerrando modal' + (shouldReload ? ' con recarga' : ''));
            
            this.modal.removeClass('show');
            $('body').css('overflow', '');
            
            // Clear iframe after animation
            setTimeout(() => {
                this.iframe.attr('src', 'about:blank');
                this.loading.removeClass('hidden');
            }, 300);

            console.log('Modal cerrado');
            
            // Reload page to reset checkout state (only if requested)
            if (shouldReload) {
                console.log('Recargando página para resetear estado del checkout');
                setTimeout(() => {
                    window.location.reload();
                }, 500);
            }
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        FrakmentaModal.init();
        
        // Expose to window for external access
        window.FrakmentaModal = FrakmentaModal;
        
        console.log('FrakmentaModal disponible globalmente');
    });

})(jQuery);
