/**
 * Frakmenta payment method for WooCommerce Blocks
 */

// Prevent multiple executions
if (window.frakmentaBlocksLoaded) {
    console.warn('frakmenta-blocks.js ya fue cargado, saltando ejecución duplicada');
} else {
    window.frakmentaBlocksLoaded = true;
    console.log('FRAKMENTA BLOCKS SCRIPT CARGADO (primera vez)');

const { registerPaymentMethod } = window.wc.wcBlocksRegistry;
const { createElement } = window.wp.element;
const { __ } = window.wp.i18n;
const { decodeEntities } = window.wp.htmlEntities;
const { getSetting } = window.wc.wcSettings;

const settings = getSetting('frakmenta_data', {});

console.log('Settings recibidos:', settings);

const label = decodeEntities(settings.title || 'Frakmenta');
const description = decodeEntities(settings.description || '');

/**
 * Content component for Frakmenta
 */
const Content = (props) => {
    console.log('Renderizando Content de Frakmenta en Blocks');
    
    const cartTotal = settings.cart_total || 0;
    const minAmount = settings.min_amount || 59;
    const maxAmount = settings.max_amount || 1000;
    
    // Use effect to call simulator after render
    const { useEffect } = window.wp.element;
    
    useEffect(() => {
        console.log('Efecto ejecutado - inicializando simulador Frakmenta');
        
        // Initialize Frakmenta global variables from settings
        if (!settings.frakmenta_url || !settings.public_key) {
            console.error('Configuración Frakmenta incompleta en settings');
            return;
        }
        
        window.fkApiKey = settings.public_key;
        window.fkApiUrl = settings.frakmenta_url;
        window.fkEcommerceUrl = settings.frakmenta_url;
        window.logoFrakmenta = settings.logo_url;
        
        console.log('Variables globales Frakmenta inicializadas desde settings');
        
        // Function to initialize simulator
        const initSimulator = () => {
            const widgetElement = document.getElementById('fk-widget-installments');
            
            if (!widgetElement) {
                console.error('Elemento #fk-widget-installments NO encontrado');
                return false;
            }
            
            if (typeof simulator !== 'function') {
                console.error('Función simulator() NO disponible todavía');
                return false;
            }
            
            const priceInCents = parseInt(widgetElement.getAttribute('data-product_price') || '0');
            const priceInEuros = priceInCents / 100;
            
            console.log('Elemento widget y función simulator() disponibles. Precio:', priceInEuros, '€');
            
            // Call simulator directly (it handles limits check internally via frakmenta_init)
            try {
                console.log('Llamando a simulator()...');
                simulator();
                console.log('simulator() ejecutado correctamente');
                return true;
            } catch (error) {
                console.error('Error al ejecutar simulator():', error);
                console.error('Stack:', error.stack);
                return false;
            }
        };
        
        // Retry mechanism: try multiple times until simulator is available
        let attempts = 0;
        const maxAttempts = 10;
        const retryInterval = 300;
        
        const tryInit = () => {
            attempts++;
            console.log(`Intento ${attempts}/${maxAttempts} de inicializar simulator()`);
            
            if (initSimulator()) {
                console.log('Simulador inicializado exitosamente');
                return;
            }
            
            if (attempts < maxAttempts) {
                setTimeout(tryInit, retryInterval);
            } else {
                console.error('No se pudo inicializar el simulador después de', maxAttempts, 'intentos');
            }
        };
        
        // Start trying after a short delay
        setTimeout(tryInit, 200);
        
    }, [cartTotal]); // Re-run when cart total changes
    
    let message = '';
    let messageStyle = {};
    
    if (cartTotal < minAmount) {
        const remaining = (minAmount - cartTotal).toFixed(2);
        //message = `Te faltan ${remaining}€ para pagar a plazos con Frakmenta`;
        //messageStyle = { color: '#ff9800', fontWeight: 'normal', fontSize: '0.9em', marginTop: '0.5em' };
    } else if (cartTotal > maxAmount) {
        message = `El importe supera el máximo de ${maxAmount}€`;
        messageStyle = { color: '#f44336', fontWeight: 'normal', fontSize: '0.9em', marginTop: '0.5em' };
    } else {
        // Dentro de límites - mostrar simulador en lugar de texto
        const elements = [
            createElement('div', { 
                key: 'widget',
                className: 'fk-installments',
                id: 'fk-widget-installments',
                'data-product_price': (cartTotal * 100).toString()
            })
        ];
        
        if (description) {
            elements.push(createElement('p', { key: 'desc', style: { marginTop: '0.5em', fontSize: '0.9em', color: '#666' } }, description));
        }
        
        return createElement(
            'div',
            { 
                className: 'frakmenta-blocks-payment-method'
            },
            elements
        );
    }
    
    // Fuera de límites - mostrar mensaje
    const elements = [
        createElement('p', { key: 'message', style: messageStyle }, message)
    ];
    
    if (description) {
        elements.push(createElement('p', { key: 'desc', style: { marginTop: '0.5em', fontSize: '0.9em', color: '#666' } }, description));
    }
    
    return createElement(
        'div',
        { 
            className: 'frakmenta-blocks-payment-method'
        },
        elements
    );
};

/**
 * Label component for Frakmenta
 */
const Label = (props) => {
    console.log('Renderizando Label de Frakmenta en Blocks');
    
    const logoUrl = settings.logo_url || '';
    
    return createElement(
        'span',
        { className: 'wc-block-components-payment-method-label', style: { display: 'flex', alignItems: 'center', gap: '10px' } },
        [
            createElement('span', { key: 'label' }, label),
            logoUrl && createElement('img', {
                key: 'logo',
                src: logoUrl,
                alt: 'Frakmenta',
                style: { height: '24px', width: 'auto' }
            })
        ]
    );
};

/**
 * Frakmenta payment method configuration
 */
const FrakmentaPaymentMethod = {
    name: 'frakmenta',
    label: createElement(Label),
    content: createElement(Content),
    edit: createElement(Content),
    canMakePayment: () => {
        console.log('Blocks: canMakePayment() llamado');
        return true;
    },
    ariaLabel: label,
    supports: {
        features: settings.supports || ['products'],
    },
};

console.log('Registrando método de pago Frakmenta para Blocks');
registerPaymentMethod(FrakmentaPaymentMethod);
console.log('Frakmenta registrado en wc.wcBlocksRegistry');

// Intercept payment processing BEFORE redirect happens
if (window.wp && window.wp.data) {
    const { select, subscribe, dispatch } = window.wp.data;
    
    console.log('Configurando interceptor de pago para Frakmenta');
    
    let isProcessing = false;
    let previousStatus = '';
    let frakmentaRedirectUrl = null;
    
    // Prevent navigation when Frakmenta redirect is detected
    window.addEventListener('beforeunload', function(e) {
        if (frakmentaRedirectUrl) {
            console.log('beforeunload: Bloqueando navegación para abrir modal');
            e.preventDefault();
            e.returnValue = '';
            
            if (window.FrakmentaModal) {
                setTimeout(() => {
                    window.FrakmentaModal.open(frakmentaRedirectUrl);
                    frakmentaRedirectUrl = null;
                }, 100);
            }
            return '';
        }
    });
    
    subscribe(() => {
        const checkoutStore = select('wc/store/checkout');
        if (!checkoutStore) return;
        
        const status = checkoutStore.getCheckoutStatus();
        const paymentStore = select('wc/store/payment');
        const activePaymentMethod = paymentStore ? paymentStore.getActivePaymentMethod() : null;
        
        // Only intercept for Frakmenta
        if (activePaymentMethod !== 'frakmenta') {
            frakmentaRedirectUrl = null;
            isProcessing = false;
            return;
        }
        
        // Reset processing flag if checkout fails or is idle
        if (status === 'idle' || status === 'failed') {
            isProcessing = false;
        }
        
        // Check if checkout just completed
        if (status === 'complete' && previousStatus !== 'complete' && !isProcessing) {
            isProcessing = true;
            console.log('Checkout completado con Frakmenta - interceptando redirect');
            
            // Get redirect URL
            const redirectUrl = checkoutStore.getRedirectUrl();
            console.log('URL de redirección:', redirectUrl);
            
            // Si no hay URL de redirección, el pago falló en el servidor
            if (!redirectUrl) {
                console.error('No hay URL de redirección - el pago falló en el servidor');
                isProcessing = false;
                return;
            }
            
            if (redirectUrl && redirectUrl.includes('/op/ecommerce/')) {
                console.log('URL de Frakmenta detectada - bloqueando redirect');
                
                // Store URL for beforeunload handler
                frakmentaRedirectUrl = redirectUrl;
                
                // Try to clear the redirect URL in the store to prevent navigation
                try {
                    const checkoutDispatch = dispatch('wc/store/checkout');
                    if (checkoutDispatch && checkoutDispatch.__internalSetRedirectUrl) {
                        checkoutDispatch.__internalSetRedirectUrl('');
                        console.log('Redirect URL limpiada del store');
                        frakmentaRedirectUrl = null; // Clear since we prevented it at source
                    }
                } catch (e) {
                    console.warn('No se pudo limpiar redirect URL del store:', e.message);
                }
                
                // Open modal immediately
                if (window.FrakmentaModal) {
                    console.log('Abriendo modal de pago');
                    setTimeout(() => {
                        window.FrakmentaModal.open(redirectUrl);
                    }, 100);
                } else {
                    console.error('FrakmentaModal no disponible - usando redirect normal');
                    frakmentaRedirectUrl = null; // Allow redirect
                }
            }
        }
        
        previousStatus = status;
    });
    
    console.log('Interceptor configurado correctamente');
}

} // End of window.frakmentaBlocksLoaded check
