<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

namespace Frakmenta\Service;

use Cart;
use Customer;
use Address;
use Country;
use Context;
use Configuration;
use Frakmenta\Config\FrakmentaConstants;
use Frakmenta\Repository\FrakmentaTransactionRepository;
use Frakmenta\Exception\FrakmentaApiException;

/**
 * Class FrakmentaTransactionService
 * 
 * Servicio para gestionar las transacciones de Frakmenta
 * Maneja la lógica de negocio relacionada con operaciones de pago
 * 
 * @package Frakmenta\Service
 */
class FrakmentaTransactionService
{
    /**
     * @var FrakmentaApiClient Cliente de la API
     */
    private FrakmentaApiClient $apiClient;
    
    /**
     * @var FrakmentaTransactionRepository Repositorio de transacciones
     */
    private FrakmentaTransactionRepository $repository;
    
    /**
     * @var string ID del comercio
     */
    private string $merchantId;
    
    /**
     * Constructor del servicio
     *
     * @param FrakmentaApiClient $apiClient Cliente de la API
     * @param FrakmentaTransactionRepository $repository Repositorio de transacciones
     * @param string $merchantId ID del comercio
     */
    public function __construct(
        FrakmentaApiClient $apiClient,
        FrakmentaTransactionRepository $repository,
        string $merchantId
    ) {
        $this->apiClient = $apiClient;
        $this->repository = $repository;
        $this->merchantId = $merchantId;
    }
    
    /**
     * Crea una nueva operación de pago en Frakmenta
     *
     * @param Cart $cart Carrito de compras
     * @param Customer $customer Cliente
     * @return array Respuesta de la API con la URL del token
     * 
     * @throws FrakmentaApiException Si hay un error en la creación
     */
    public function createOperation(Cart $cart, Customer $customer): array
    {
        \PrestaShopLogger::addLog(
            'FrakmentaTransactionService: Iniciando createOperation - Cart: ' . $cart->id . ' | Customer: ' . $customer->id,
            1
        );
        
        $address = new Address((int) $cart->id_address_invoice);
        $country = new Country((int) $address->id_country);
        
        \PrestaShopLogger::addLog(
            'FrakmentaTransactionService: Address ID: ' . $address->id . ' | Country: ' . $country->iso_code,
            1
        );
        
        $operationData = $this->buildOperationData($cart, $customer, $address, $country);
        
        \PrestaShopLogger::addLog(
            'FrakmentaTransactionService: Operation data construida - Invoice: ' . $operationData['invoice_id'],
            1
        );
        
        // Log de datos completos enviados a la API (JSON RAW)
        \PrestaShopLogger::addLog(
            'FrakmentaTransactionService: DATOS ENVIADOS A API (JSON): ' . json_encode($operationData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE),
            1
        );
        
        try {
            \PrestaShopLogger::addLog(
                'FrakmentaTransactionService: Llamando a API createOperation...',
                1
            );
            
            $response = $this->apiClient->createOperation($operationData);
            
            \PrestaShopLogger::addLog(
                'FrakmentaTransactionService: Respuesta API recibida - Status: ' . ($response['status'] ?? 'unknown'),
                1
            );
            
            // Guardar la transacción en base de datos
            $saved = $this->saveTransaction($cart, $customer, $response, $operationData['invoice_id'], $operationData['product_price']);
            
            \PrestaShopLogger::addLog(
                'FrakmentaTransactionService: Transacción guardada en DB: ' . ($saved ? 'SI' : 'NO'),
                $saved ? 1 : 2
            );
            
            return $response;
            
        } catch (FrakmentaApiException $e) {
            \PrestaShopLogger::addLog(
                'FrakmentaTransactionService: Error API - ' . $e->getMessage(),
                3
            );
            $this->handleOperationError($e, $operationData);
            throw $e;
        }
    }
    
    /**
     * Construye los datos de la operación
     *
     * @param Cart $cart Carrito
     * @param Customer $customer Cliente
     * @param Address $address Dirección
     * @param Country $country País
     * @return array Datos de la operación
     */
    private function buildOperationData(
        Cart $cart,
        Customer $customer,
        Address $address,
        Country $country
    ): array {
        $products = $cart->getProducts(true);
        $invoiceId = $this->generateInvoiceId($cart->id);
        
        // Obtener total y convertir a centavos
        $totalDecimal = $cart->getOrderTotal(true, Cart::BOTH);
        $productPrice = (int) ($totalDecimal * 100);
        
        \PrestaShopLogger::addLog(
            'FrakmentaTransactionService: PRECIO - Decimal: ' . $totalDecimal . ' | Centavos (int): ' . $productPrice . ' | Type: ' . gettype($productPrice),
            1
        );
        
        return [
            'merchant_id' => $this->merchantId,
            'invoice_id' => $invoiceId,
            'product_price' => $productPrice,
            'currency_code' => FrakmentaConstants::DEFAULT_CURRENCY,
            'delegation' => FrakmentaConstants::DEFAULT_DELEGATION,
            'type' => FrakmentaConstants::OPERATION_TYPE,
            'customer' => $this->buildCustomerData($customer, $address, $country),
            'order' => $this->buildOrderData($cart, $products),
            'flow_config' => $this->buildFlowConfig($cart, $customer, $invoiceId),
            'other_data' => $this->buildOtherData()
        ];
    }
    
    /**
     * Construye los datos del cliente
     *
     * @param Customer $customer Cliente
     * @param Address $address Dirección
     * @param Country $country País
     * @return array Datos del cliente
     */
    private function buildCustomerData(
        Customer $customer,
        Address $address,
        Country $country
    ): array {
        $dateJoined = date_create($customer->date_add);
        
        return [
            'identification' => [
                'nif' => $address->dni ?? '',
                'legal_first_name' => $address->firstname ?: $customer->firstname,
                'legal_last_name' => $address->lastname ?: $customer->lastname,
                'date_of_birth' => $customer->birthday,
                'mobile_phone_number' => $address->phone_mobile ?: $address->phone,
                'email' => $customer->email
            ],
            'address' => [
                'line_1' => $address->address1,
                'line_2' => $address->address2 ?: ' ',
                'phone' => $address->phone,
                'city' => $address->city,
                'state' => $address->city,
                'county' => $country->name[1],
                'country_code' => $country->iso_code,
                'postcode' => $address->postcode,
            ],
            'store_details' => [
                'customer_date_joined' => $dateJoined->format('Y-m-d'),
                'customer_last_login' => $dateJoined->format('Y-m-d')
            ],
            'financial' => [
                'salary' => 0,
                'currency' => FrakmentaConstants::DEFAULT_CURRENCY,
                'employment_status' => 'N/A',
                'contract_type' => 'N/A'
            ],
            'other_data' => [
                ['name' => 'Tienda', 'type' => 'STRING', 'value' => Configuration::get('PS_SHOP_NAME')],
                ['name' => 'Ecommerce', 'type' => 'STRING', 'value' => 'PRESTASHOP'],
                ['name' => 'Version', 'type' => 'STRING', 'value' => _PS_VERSION_],
                ['name' => 'Enviroment', 'type' => 'STRING', 
                 'value' => Configuration::get('FRAKMENTA_MODE') == 0 ? 'TEST' : 'PRODUCTION']
            ]
        ];
    }
    
    /**
     * Construye los datos del pedido
     *
     * @param Cart $cart Carrito
     * @param array $products Productos del carrito
     * @return array Datos del pedido
     */
    private function buildOrderData(Cart $cart, array $products): array
    {
        $cartProducts = [];
        
        foreach ($products as $product) {
            $cartProducts[] = [
                'id' => (string) $product['id_product'],
                'name' => substr($product['name'], 0, 900),
                'quantity' => $product['cart_quantity'],
                'price' => (float) number_format($product['price'], 2, '.', ''),
                'tax_rate' => (float) $product['rate'],
                'description' => strip_tags(substr($product['description_short'], 0, 890)),
                'url' => 'https://frakmenta.com',
                'image_url' => 'https://frakmenta.com'
            ];
        }
        
        return [
            'id' => (string) $cart->id,
            'products' => $cartProducts
        ];
    }
    
    /**
     * Construye la configuración del flujo
     *
     * @param Cart $cart Carrito
     * @param Customer $customer Cliente
     * @param string $invoiceId ID de la factura
     * @return array Configuración del flujo
     */
    private function buildFlowConfig(Cart $cart, Customer $customer, string $invoiceId): array
    {
        $context = Context::getContext();
        
        $successUrl = $context->link->getModuleLink(
            'frakmenta',
            'success'
        ) . '?id=' . $customer->secure_key . 
          '&cart=' . $cart->id . 
          '&invoice=' . $invoiceId;
        
        $notificationUrl = $context->link->getModuleLink('frakmenta', 'notifications');
        
        // Ajustar protocolo si es necesario
        if (_PS_BASE_URL_ == 'http' && Configuration::get('FRAKMENTA_MODE') == 0) {
            $successUrl = str_replace('https', 'http', $successUrl);
            $notificationUrl = str_replace('https', 'http', $notificationUrl);
        }
        
        $baseUri = Context::getContext()->shop->getBaseURI();
        $koUrl = _PS_BASE_URL_ . $baseUri . 'index.php?controller=order&step=3&fk=KO';
        
        \PrestaShopLogger::addLog(
            'FrakmentaTransactionService: URLs construidas - Success: ' . $successUrl . ' | KO: ' . $koUrl,
            1
        );
        
        return [
            'success_url' => $successUrl,
            'notification_url' => $notificationUrl,
            'ko_url' => $koUrl,
        ];
    }
    
    /**
     * Construye datos adicionales
     *
     * @return array Datos adicionales
     */
    private function buildOtherData(): array
    {
        return [
            ['name' => 'N/A', 'type' => 'STRING', 'value' => 'N/A']
        ];
    }
    
    /**
     * Genera un ID único para la factura
     *
     * @param int $cartId ID del carrito
     * @return string ID de factura generado
     */
    private function generateInvoiceId(int $cartId): string
    {
        return hash('sha256', $this->merchantId . '-' . $cartId . '-' . date('YmdHis'));
    }
    
    /**
     * Guarda la transacción en base de datos
     *
     * @param Cart $cart Carrito
     * @param Customer $customer Cliente
     * @param array $response Respuesta de la API
     * @param string $invoiceId ID de la factura
     * @param int $amount Monto en centavos
     * @return bool True si se guardó correctamente
     */
    private function saveTransaction(
        Cart $cart,
        Customer $customer,
        array $response,
        string $invoiceId,
        int $amount
    ): bool {
        $state = (strtoupper($response['status']) === 'OK') 
            ? FrakmentaConstants::STATE_INITIALIZED 
            : FrakmentaConstants::STATE_FAILED;
            
        $operationId = (strtoupper($response['status']) === 'OK') 
            ? $response['data']['operation_id'] 
            : -1;
        
        return $this->repository->create([
            'id_cart' => $cart->id,
            'id_custom_client' => $customer->secure_key,
            'id_operation' => $operationId,
            'id_invoice' => $invoiceId,
            'ammount' => $amount,
            'petition_state' => $state
        ]);
    }
    
    /**
     * Obtiene el estado de una operación
     *
     * @param int $operationId ID de la operación
     * @return array Estado de la operación
     * 
     * @throws FrakmentaApiException Si hay un error
     */
    public function getOperationStatus(int $operationId): array
    {
        return $this->apiClient->getOperationStatus($this->merchantId, $operationId);
    }
    
    /**
     * Maneja errores de operaciones
     *
     * @param FrakmentaApiException $exception Excepción capturada
     * @param array $operationData Datos de la operación
     * @return void
     */
    private function handleOperationError(
        FrakmentaApiException $exception,
        array $operationData
    ): void {
        $mode = Configuration::get('FRAKMENTA_MODE') == 0 ? 'Pruebas' : 'Producción';
        
        $message = '<b>Información del comercio</b><br><br>';
        $message .= '<b>Tienda:</b> ' . Configuration::get('PS_SHOP_NAME') . '<br>';
        $message .= '<b>Conexión:</b> ' . $mode . '<br>';
        $message .= '<b>Error:</b> ' . $exception->getMessage() . '<br>';
        $message .= '<b>Código HTTP:</b> ' . $exception->getHttpCode() . '<br>';
        
        if ($exception->getResponseData()) {
            $message .= '<b>Respuesta API:</b> <pre>' . 
                       print_r($exception->getResponseData(), true) . '</pre>';
        }
        
        $message .= '<b>Request:</b> <pre>' . print_r($operationData, true) . '</pre>';
        
        // Log el error
        \PrestaShopLogger::addLog(
            'Frakmenta operation error: ' . $exception->getMessage(),
            3,
            null,
            'Cart',
            $operationData['order']['id'] ?? 0
        );
        
        // Enviar email de notificación
        $this->sendErrorEmail($message);
    }
    
    /**
     * Envía un email de error
     *
     * @param string $message Mensaje de error
     * @return void
     */
    private function sendErrorEmail(string $message): void
    {
        \Mail::Send(
            (int) Configuration::get('PS_LANG_DEFAULT'),
            'contact',
            'Error en transacción Frakmenta - ' . _PS_VERSION_,
            [
                '{email}' => Configuration::get('PS_SHOP_EMAIL'),
                '{message}' => $message
            ],
            'desarrollo-frakmenta@findirect.com',
            null,
            null,
            null
        );
    }
}
