<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

namespace Frakmenta\Service;

use Configuration;
use Tools;
use Frakmenta\Config\FrakmentaConstants;

/**
 * Class FrakmentaConfigService
 * 
 * Servicio para gestionar la configuración del módulo
 * Centraliza el acceso a la configuración y validaciones
 * 
 * @package Frakmenta\Service
 */
class FrakmentaConfigService
{
    /**
     * Obtiene la URL de la API según el modo configurado
     *
     * @return string URL de la API
     */
    public static function getApiUrl(): string
    {
        return Configuration::get('FRAKMENTA_URL') ?: FrakmentaConstants::PRODUCTION_URL;
    }
    
    /**
     * Obtiene la clave pública configurada
     *
     * @return string Clave pública
     */
    public static function getPublicKey(): string
    {
        return Configuration::get('FRAKMENTA_PUBLIC_KEY') ?: '';
    }
    
    /**
     * Obtiene la clave privada configurada
     *
     * @return string Clave privada
     */
    public static function getPrivateKey(): string
    {
        return Configuration::get('FRAKMENTA_PRIVATE_KEY') ?: '';
    }
    
    /**
     * Obtiene el ID del comercio
     *
     * @return string ID del comercio
     */
    public static function getMerchantId(): string
    {
        return Configuration::get('FRAKMENTA_MERCHANT_ID') ?: '';
    }
    
    /**
     * Obtiene el modo de operación (test/producción)
     *
     * @return int Modo de operación
     */
    public static function getMode(): int
    {
        return (int) Configuration::get('FRAKMENTA_MODE');
    }
    
    /**
     * Verifica si está en modo test
     *
     * @return bool True si está en modo test
     */
    public static function isTestMode(): bool
    {
        return self::getMode() === FrakmentaConstants::MODE_TEST;
    }
    
    /**
     * Verifica si está en modo producción
     *
     * @return bool True si está en modo producción
     */
    public static function isProductionMode(): bool
    {
        return self::getMode() === FrakmentaConstants::MODE_PRODUCTION;
    }
    
    /**
     * Verifica si el simulador de productos está activado
     *
     * @return bool True si está activado
     */
    public static function isProductSimulatorEnabled(): bool
    {
        return (int) Configuration::get('FRAKMENTA_PRODUCT_OPTION') === 1;
    }
    
    /**
     * Obtiene la ubicación del simulador en el DOM
     *
     * @return string Selector CSS
     */
    public static function getSimulatorLocation(): string
    {
        $location = Configuration::get('FRAKMENTA_LOCATION_SIMULATOR');
        return $location ?: FrakmentaConstants::LOCATION_SIMULATOR_DEFAULT;
    }
    
    /**
     * Guarda la configuración en modo test
     *
     * @param array $data Datos del formulario
     * @return bool True si se guardó correctamente
     */
    public static function saveTestConfiguration(array $data): bool
    {
        Configuration::updateValue('FRAKMENTA_ACCOUNT', Tools::getValue('fk_account'));
        Configuration::updateValue('FRAKMENTA_URL', FrakmentaConstants::TEST_URL);
        Configuration::updateValue('FRAKMENTA_PUBLIC_KEY', FrakmentaConstants::TEST_PUBLIC_KEY);
        Configuration::updateValue('FRAKMENTA_PRIVATE_KEY', FrakmentaConstants::TEST_PRIVATE_KEY);
        Configuration::updateValue('FRAKMENTA_MERCHANT_ID', FrakmentaConstants::TEST_MERCHANT_ID);
        Configuration::updateValue('FRAKMENTA_MODE', FrakmentaConstants::MODE_TEST);
        Configuration::updateValue('FRAKMENTA_PRODUCT_OPTION', Tools::getValue('fk_sim_product'));
        Configuration::updateValue('FRAKMENTA_LOCATION_SIMULATOR', Tools::getValue('fk_location_simulator'));
        Configuration::updateValue('FRAKMENTA_DELEGATION', FrakmentaConstants::DEFAULT_DELEGATION);
        
        return true;
    }
    
    /**
     * Guarda la configuración en modo producción
     *
     * @param array $data Datos del formulario
     * @return bool True si se guardó correctamente
     */
    public static function saveProductionConfiguration(array $data): bool
    {
        Configuration::updateValue('FRAKMENTA_URL', FrakmentaConstants::PRODUCTION_URL);
        Configuration::updateValue('FRAKMENTA_PUBLIC_KEY', Tools::getValue('fk_public_key'));
        Configuration::updateValue('FRAKMENTA_PRIVATE_KEY', Tools::getValue('fk_private_key'));
        Configuration::updateValue('FRAKMENTA_MERCHANT_ID', Tools::getValue('fk_merchant_id'));
        Configuration::updateValue('FRAKMENTA_MODE', FrakmentaConstants::MODE_PRODUCTION);
        Configuration::updateValue('FRAKMENTA_ACCOUNT', Tools::getValue('fk_account'));
        Configuration::updateValue('FRAKMENTA_PRODUCT_OPTION', Tools::getValue('fk_sim_product'));
        Configuration::updateValue('FRAKMENTA_LOCATION_SIMULATOR', Tools::getValue('fk_location_simulator'));
        Configuration::updateValue('FRAKMENTA_DELEGATION', FrakmentaConstants::DEFAULT_DELEGATION);
        
        return true;
    }
    
    /**
     * Valida que la configuración sea correcta
     *
     * @return array Array con errores encontrados (vacío si no hay errores)
     */
    public static function validateConfiguration(): array
    {
        $errors = [];
        
        if (empty(self::getPublicKey())) {
            $errors[] = 'La clave pública es requerida';
        }
        
        if (empty(self::getPrivateKey())) {
            $errors[] = 'La clave privada es requerida';
        }
        
        if (empty(self::getMerchantId())) {
            $errors[] = 'El ID de comercio es requerido';
        }
        
        return $errors;
    }
    
    /**
     * Verifica si el módulo está configurado correctamente
     *
     * @return bool True si está configurado
     */
    public static function isConfigured(): bool
    {
        return !empty(self::getPublicKey()) && 
               !empty(self::getPrivateKey()) && 
               !empty(self::getMerchantId());
    }
    
    /**
     * Habilita iframes en PrestaShop si no están habilitados
     *
     * @return void
     */
    public static function enableIframes(): void
    {
        if (empty(Configuration::get('PS_ALLOW_HTML_IFRAME'))) {
            Configuration::updateValue('PS_ALLOW_HTML_IFRAME', 1);
        }
    }
    
    /**
     * Verifica si SSL está habilitado
     *
     * @return bool True si SSL está habilitado
     */
    public static function isSslEnabled(): bool
    {
        return (bool) Configuration::get('PS_SSL_ENABLED');
    }
    
    /**
     * Verifica si SSL está habilitado en todas partes
     *
     * @return bool True si SSL está habilitado en todas partes
     */
    public static function isSslEnabledEverywhere(): bool
    {
        return (bool) Configuration::get('PS_SSL_ENABLED_EVERYWHERE');
    }
    
    /**
     * Elimina toda la configuración del módulo
     *
     * @return bool True si se eliminó correctamente
     */
    public static function deleteConfiguration(): bool
    {
        $keys = [
            'FRAKMENTA_ACCOUNT',
            'FRAKMENTA_URL',
            'FRAKMENTA_PUBLIC_KEY',
            'FRAKMENTA_PRIVATE_KEY',
            'FRAKMENTA_MERCHANT_ID',
            'FRAKMENTA_MODE',
            'FRAKMENTA_PRODUCT_OPTION',
            'FRAKMENTA_LOCATION_SIMULATOR',
            'FRAKMENTA_DELEGATION',
        ];
        
        foreach ($keys as $key) {
            Configuration::deleteByName($key);
        }
        
        return true;
    }
}
