<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

namespace Frakmenta\Service;

use Context;
use Configuration;
use Media;

/**
 * Class FrakmentaAssetService
 * 
 * Servicio para gestionar la carga de assets (CSS, JS)
 * Elimina la duplicación de código en los diferentes hooks
 * 
 * @package Frakmenta\Service
 */
class FrakmentaAssetService
{
    /**
     * @var string Nombre del módulo
     */
    private string $moduleName;
    
    /**
     * @var Context Contexto de PrestaShop
     */
    private Context $context;
    
    /**
     * @var string Versión del módulo
     */
    private string $version;
    
    /**
     * Constructor del servicio
     *
     * @param string $moduleName Nombre del módulo
     * @param string $version Versión del módulo
     * @param Context|null $context Contexto de PrestaShop
     */
    public function __construct(
        string $moduleName,
        string $version,
        ?Context $context = null
    ) {
        $this->moduleName = $moduleName;
        $this->version = $version;
        $this->context = $context ?: Context::getContext();
    }
    
    /**
     * Registra todos los assets de Frakmenta
     * Elimina código duplicado de hookDisplayHeader, hookHeader, hookActionFrontControllerSetMedia
     *
     * @return void
     */
    public function registerFrakmentaAssets(): void
    {
        $this->registerStylesheets();
        $this->registerJavascripts();
        $this->registerMediaDefinitions();
    }
    
    /**
     * Registra las hojas de estilo
     *
     * @return void
     */
    private function registerStylesheets(): void
    {
        // Estilos locales
        $localStylesheets = [
            $this->moduleName . '-frakmenta' => 
                'modules/' . $this->moduleName . '/views/css/frakmenta.css',
            $this->moduleName . '-frakmenta-iframe' => 
                'modules/' . $this->moduleName . '/views/css/iframe-ecommerce.min.css',
        ];
        
        foreach ($localStylesheets as $id => $path) {
            $this->context->controller->registerStylesheet($id, $path);
        }
        
        // Estilo remoto de Frakmenta
        $this->context->controller->registerStylesheet(
            $this->moduleName . '-widget-ecommerce',
            Configuration::get('FRAKMENTA_URL') . '/css/widget-ecommerce.css',
            [
                'media' => 'all',
                'priority' => 200,
                'server' => 'remote'
            ]
        );
    }
    
    /**
     * Registra los archivos JavaScript
     *
     * @return void
     */
    private function registerJavascripts(): void
    {
        // Scripts locales
        $localScripts = [
            $this->moduleName . '-operation-frakmenta' => [
                'path' => 'modules/' . $this->moduleName . '/views/js/frakmenta_front_operation.js',
                'options' => []
            ],
            $this->moduleName . '-validations-frakmenta' => [
                'path' => 'modules/' . $this->moduleName . '/views/js/frakmenta_front_validations.js',
                'options' => ['position' => 'head']
            ],
        ];
        
        foreach ($localScripts as $id => $config) {
            $this->context->controller->registerJavascript(
                $id,
                $config['path'],
                $config['options']
            );
        }
        
        // Script remoto de Frakmenta
        $this->context->controller->registerJavascript(
            $this->moduleName . '-widgetEcommerce',
            Configuration::get('FRAKMENTA_URL') . '/js/widgetEcommerce.js',
            [
                'server' => 'remote',
                'priority' => 200,
                'position' => 'head'
            ]
        );
    }
    
    /**
     * Registra las definiciones JavaScript
     *
     * @return void
     */
    private function registerMediaDefinitions(): void
    {
        Media::addJsDef([
            'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiKey' => Configuration::get('FRAKMENTA_PUBLIC_KEY'),
            'FrakmentaUrlGetToken' => $this->context->link->getModuleLink(
                $this->moduleName,
                'payment',
                ['ajax' => true]
            ),
            'fkPlatform' => 'prestashop',
            'fkPlatformVersion' => _PS_VERSION_
        ]);
    }
    
    /**
     * Registra los assets para el simulador de productos
     *
     * @param int $productPrice Precio del producto en centavos
     * @param string $position Posición en el DOM donde se mostrará el simulador
     * @return void
     */
    public function registerProductSimulatorAssets(
        int $productPrice,
        string $position = 'bottom'
    ): void {
        Media::addJsDef([
            'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiKey' => Configuration::get('FRAKMENTA_PUBLIC_KEY'),
            'productPrice' => $productPrice,
            'fkLocationSimulador' => Configuration::get('FRAKMENTA_LOCATION_SIMULATOR') ?: '.product_attributes',
            'logoFrakmenta' => $this->context->shop->getBaseURI() .
                                'modules/' . $this->moduleName . '/frakmenta-logo-original.png',
            'fkPlatform' => 'prestashop',
            'fkPlatformVersion' => _PS_VERSION_
        ]);
        
        $this->context->controller->registerStylesheet(
            $this->moduleName . '-widget-ecommerce',
            Configuration::get('FRAKMENTA_URL') . '/css/widget-ecommerce.css',
            [
                'media' => 'all',
                'priority' => 200,
                'server' => 'remote'
            ]
        );
        
        $this->context->controller->registerJavascript(
            $this->moduleName . '-widgetEcommerce',
            Configuration::get('FRAKMENTA_URL') . '/js/widgetEcommerce.js',
            [
                'server' => 'remote',
                'priority' => 200,
                'position' => 'head'
            ]
        );
        
        $this->context->controller->registerJavascript(
            $this->moduleName . '-fk-simulator-prod',
            'modules/' . $this->moduleName . '/views/js/frakmenta_front_products.js',
            [
                'server' => 'local',
                'priority' => 200,
                'position' => $position
            ]
        );
    }
    
    /**
     * Registra assets para módulos AMP
     *
     * @return array Array con recursos y estilos
     */
    public function getAmpAssets(): array
    {
        $base = _PS_BASE_URL_;
        if (Configuration::get('PS_SSL_ENABLED') == 1) {
            $base = str_replace('http', 'https', _PS_BASE_URL_);
        }
        
        $resources = [
            'https://static.frakmenta.com/js/jquery.min.js',
            Configuration::get('FRAKMENTA_URL') . '/js/widgetEcommerce.js?v=' . $this->version,
            $base . __PS_BASE_URI__ . 'modules/' . $this->moduleName . 
                '/views/js/frakmenta_front_operation.js?v=' . $this->version,
            $base . __PS_BASE_URI__ . 'modules/' . $this->moduleName . 
                '/views/js/frakmenta_front_validations.js?v=' . $this->version,
        ];
        
        $styles = [
            $base . __PS_BASE_URI__ . 'modules/' . $this->moduleName . 
                '/views/css/frakmenta.css?v=' . $this->version,
            $base . __PS_BASE_URI__ . 'modules/' . $this->moduleName . 
                '/views/css/iframe-ecommerce.min.css?v=' . $this->version,
            Configuration::get('FRAKMENTA_URL') . '/css/widget-ecommerce.css?v=' . $this->version,
        ];
        
        return [
            'resources' => $resources,
            'styles' => $styles,
            'variables' => [
                'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
                'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
                'fkApiKey' => Configuration::get('FRAKMENTA_PUBLIC_KEY'),
                'FrakmentaUrlGetToken' => $this->context->link->getModuleLink(
                    $this->moduleName,
                    'payment',
                    ['ajax' => true]
                ),
                'fkPlatform' => 'prestashop',
                'fkPlatformVersion' => _PS_VERSION_
            ]
        ];
    }
    
    /**
     * Registra assets del back office
     *
     * @param string $merchantId ID del comercio
     * @param string $publicKey Clave pública
     * @param string $signature Firma de validación
     * @return void
     */
    public function registerBackOfficeAssets(
        string $merchantId,
        string $publicKey,
        string $signature
    ): void {
        Media::addJsDef([
            'frakmenta_merchant_id_validation' => $merchantId,
            'frakmenta_public_key_validation' => $publicKey,
            'frakmenta_sign_validation' => $signature,
        ]);
        
        if (version_compare(_PS_VERSION_, '1.5', '>=')) {
            $this->context->controller->addJquery();
            $this->context->controller->addJQueryPlugin('fancybox');
            $this->context->controller->addCSS(
                _PS_MODULE_DIR_ . $this->moduleName . '/views/css/frakmenta.css'
            );
        }
        
        $this->context->controller->addJS(
            _PS_MODULE_DIR_ . $this->moduleName . '/views/js/frakmenta_back_office.js'
        );
    }
}
