<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

namespace Frakmenta\Service;

use Frakmenta\Config\FrakmentaConstants;
use Frakmenta\Exception\FrakmentaApiException;

/**
 * Class FrakmentaApiClient
 * 
 * Cliente para comunicación con la API de Frakmenta
 * Maneja todas las peticiones HTTP a los endpoints de Frakmenta
 * Compatible con PHP 8.4
 * 
 * @package Frakmenta\Service
 */
class FrakmentaApiClient
{
    /**
     * @var string URL base de la API
     */
    private string $baseUrl;
    
    /**
     * @var string Clave pública de Frakmenta
     */
    private string $publicKey;
    
    /**
     * @var string Clave privada de Frakmenta
     */
    private string $privateKey;
    
    /**
     * @var int Timeout para las peticiones en segundos
     */
    private int $timeout;
    
    /**
     * Constructor de FrakmentaApiClient
     *
     * @param string $baseUrl URL base de la API
     * @param string $publicKey Clave pública
     * @param string $privateKey Clave privada
     * @param int $timeout Timeout en segundos (por defecto 30)
     * 
     * @throws \RuntimeException Si cURL no está disponible
     */
    public function __construct(
        string $baseUrl,
        string $publicKey,
        string $privateKey,
        int $timeout = FrakmentaConstants::API_TIMEOUT
    ) {
        if (!function_exists('curl_init')) {
            throw new \RuntimeException('cURL extension is required');
        }
        
        $this->baseUrl = rtrim($baseUrl, '/');
        $this->publicKey = $publicKey;
        $this->privateKey = $privateKey;
        $this->timeout = $timeout;
    }
    
    /**
     * Realiza una petición POST a la API
     *
     * @param string $endpoint Endpoint de la API
     * @param array $data Datos a enviar
     * @return array Respuesta decodificada
     * 
     * @throws FrakmentaApiException Si hay un error en la petición
     */
    public function post(string $endpoint, array $data): array
    {
        return $this->request('POST', $endpoint, $data);
    }
    
    /**
     * Realiza una petición GET a la API
     *
     * @param string $endpoint Endpoint de la API
     * @param array $params Parámetros de la query string
     * @return array Respuesta decodificada
     * 
     * @throws FrakmentaApiException Si hay un error en la petición
     */
    public function get(string $endpoint, array $params = []): array
    {
        $url = $endpoint;
        if (!empty($params)) {
            $url .= '?' . http_build_query($params);
        }
        return $this->request('GET', $url);
    }
    
    /**
     * Realiza una petición HTTP a la API
     *
     * @param string $method Método HTTP (GET, POST, etc.)
     * @param string $endpoint Endpoint de la API
     * @param array|null $data Datos a enviar (solo para POST)
     * @return array Respuesta decodificada
     * 
     * @throws FrakmentaApiException Si hay un error en la petición
     */
    private function request(string $method, string $endpoint, ?array $data = null): array
    {
        $url = $this->baseUrl . $endpoint;
        
        $curl = curl_init();
        if ($curl === false) {
            throw new FrakmentaApiException('Failed to initialize cURL');
        }
        
        $headers = ['Content-Type: application/json'];
        
        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => FrakmentaConstants::API_MAX_REDIRECTS,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => $headers,
        ];
        
        if ($method === 'POST' && $data !== null) {
            $jsonData = json_encode($data);
            if ($jsonData === false) {
                throw new FrakmentaApiException(
                    'JSON encoding error: ' . json_last_error_msg()
                );
            }
            $options[CURLOPT_POSTFIELDS] = $jsonData;
            $headers[] = 'Content-Length: ' . strlen($jsonData);
            $options[CURLOPT_HTTPHEADER] = $headers;
        }
        
        curl_setopt_array($curl, $options);
        
        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $error = curl_error($curl);
        
        curl_close($curl);
        
        if ($response === false) {
            throw new FrakmentaApiException(
                sprintf('cURL error: %s', $error)
            );
        }
        
        // Convertir encoding si es necesario (compatible PHP 8.4)
        if (mb_detect_encoding($response, 'UTF-8', true) === false) {
            $response = mb_convert_encoding($response, 'UTF-8', 'ISO-8859-1');
        }
        
        $decodedResponse = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new FrakmentaApiException(
                'JSON decode error: ' . json_last_error_msg()
            );
        }
        
        if ($httpCode >= 400) {
            throw new FrakmentaApiException(
                sprintf(
                    'API error (HTTP %d): %s',
                    $httpCode,
                    $decodedResponse['message'] ?? 'Unknown error'
                ),
                0,
                $httpCode,
                $decodedResponse
            );
        }
        
        return $decodedResponse;
    }
    
    /**
     * Valida la configuración con la API de Frakmenta
     *
     * @param string $merchantId ID del comercio
     * @return bool True si la validación es exitosa
     * 
     * @throws FrakmentaApiException Si hay un error en la validación
     */
    public function validateConfiguration(string $merchantId): bool
    {
        try {
            $signature = $this->generateValidationSignature($merchantId);
            
            $response = $this->post(FrakmentaConstants::ENDPOINT_CONFIG_VALIDATION, [
                'merchant_id' => $merchantId,
                'delegation' => FrakmentaConstants::DEFAULT_DELEGATION,
                'public_key' => $this->publicKey,
                'signature' => $signature,
            ]);
            
            return ($response['status'] ?? '') === FrakmentaConstants::STATE_OK;
            
        } catch (FrakmentaApiException $e) {
            \PrestaShopLogger::addLog(
                'Frakmenta config validation failed: ' . $e->getMessage(),
                3
            );
            return false;
        }
    }
    
    /**
     * Obtiene los límites del comercio
     *
     * @return array Datos de límites del comercio
     * 
     * @throws FrakmentaApiException Si hay un error en la petición
     */
    public function getMerchantLimits(): array
    {
        return $this->get(FrakmentaConstants::ENDPOINT_LIMITS, [
            'apikey' => $this->publicKey
        ]);
    }
    
    /**
     * Obtiene el estado de una operación
     *
     * @param string $merchantId ID del comercio
     * @param int $operationId ID de la operación
     * @return array Estado de la operación
     * 
     * @throws FrakmentaApiException Si hay un error en la petición
     */
    public function getOperationStatus(string $merchantId, int $operationId): array
    {
        $signature = hash(
            'sha256',
            implode('|', [
                $merchantId,
                FrakmentaConstants::DEFAULT_DELEGATION,
                FrakmentaConstants::OPERATION_TYPE,
                $operationId,
                $this->privateKey
            ]),
            false
        );
        
        return $this->post(FrakmentaConstants::ENDPOINT_OPERATIONS_STATUS, [
            'merchant_id' => $merchantId,
            'delegation' => FrakmentaConstants::DEFAULT_DELEGATION,
            'type' => FrakmentaConstants::OPERATION_TYPE,
            'operation_id' => $operationId,
            'signature' => $signature
        ]);
    }
    
    /**
     * Crea una nueva operación de pago
     *
     * @param array $operationData Datos de la operación
     * @return array Respuesta con los datos de la operación creada
     * 
     * @throws FrakmentaApiException Si hay un error en la creación
     */
    public function createOperation(array $operationData): array
    {
        $signature = $this->generateOperationSignature($operationData);
        $operationData['signature'] = $signature;
        
        $fullUrl = $this->baseUrl . FrakmentaConstants::ENDPOINT_OPERATIONS;
        
        \PrestaShopLogger::addLog(
            'FrakmentaApiClient: Enviando POST a: ' . $fullUrl,
            1
        );
        
        \PrestaShopLogger::addLog(
            'FrakmentaApiClient: Signature generada: ' . $signature,
            1
        );
        
        \PrestaShopLogger::addLog(
            'FrakmentaApiClient: Componentes de firma: ' . 
            $operationData['merchant_id'] . '|' . 
            $operationData['delegation'] . '|' . 
            $operationData['type'] . '|' . 
            $operationData['invoice_id'] . '|' . 
            $operationData['product_price'] . '|' . 
            $operationData['currency_code'] . '|[PRIVATE_KEY]',
            1
        );
        
        return $this->post(FrakmentaConstants::ENDPOINT_OPERATIONS, $operationData);
    }
    
    /**
     * Genera la firma para validación de configuración
     *
     * @param string $merchantId ID del comercio
     * @return string Firma generada
     */
    private function generateValidationSignature(string $merchantId): string
    {
        $parts = [
            $merchantId,
            FrakmentaConstants::DEFAULT_DELEGATION,
            $this->privateKey,
        ];
        
        return hash('sha256', implode('|', $parts));
    }
    
    /**
     * Genera la firma para una operación
     *
     * @param array $operationData Datos de la operación
     * @return string Firma generada
     */
    private function generateOperationSignature(array $operationData): string
    {
        $parts = [
            $operationData['merchant_id'],
            $operationData['delegation'],
            $operationData['type'],
            $operationData['invoice_id'],
            $operationData['product_price'],
            $operationData['currency_code'],
            $this->privateKey,
        ];
        
        return hash('sha256', implode('|', $parts), false);
    }
    
    /**
     * Obtiene la clave pública configurada
     *
     * @return string
     */
    public function getPublicKey(): string
    {
        return $this->publicKey;
    }
    
    /**
     * Obtiene la URL base configurada
     *
     * @return string
     */
    public function getBaseUrl(): string
    {
        return $this->baseUrl;
    }
}
