<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

namespace Frakmenta\Repository;

use Db;
use DbQuery;
use Frakmenta\Config\FrakmentaConstants;

/**
 * Class FrakmentaTransactionRepository
 * 
 * Repositorio para gestionar las transacciones de Frakmenta en base de datos
 * Utiliza consultas preparadas para prevenir inyección SQL
 * Compatible con PHP 8.4
 * 
 * @package Frakmenta\Repository
 */
class FrakmentaTransactionRepository
{
    /**
     * @var string Nombre de la tabla
     */
    private string $tableName;
    
    /**
     * Constructor del repositorio
     */
    public function __construct()
    {
        $this->tableName = _DB_PREFIX_ . FrakmentaConstants::TABLE_NAME;
    }
    
    /**
     * Crea una nueva transacción en la base de datos
     *
     * @param array $data Datos de la transacción
     * @return bool True si la inserción fue exitosa
     */
    public function create(array $data): bool
    {
        return Db::getInstance()->insert(
            FrakmentaConstants::TABLE_NAME,
            [
                'id_cart' => (int) $data['id_cart'],
                'id_custom_client' => pSQL($data['id_custom_client']),
                'id_operation' => (int) $data['id_operation'],
                'id_invoice' => pSQL($data['id_invoice']),
                'ammount' => (int) $data['ammount'],
                'petition_state' => pSQL($data['petition_state']),
                'attempts' => 0,
            ]
        );
    }
    
    /**
     * Busca una transacción por ID de factura
     *
     * @param string $invoiceId ID de la factura
     * @return array|null Datos de la transacción o null si no existe
     */
    public function findByInvoiceId(string $invoiceId): ?array
    {
        $query = new DbQuery();
        $query->select('*')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_invoice = "' . pSQL($invoiceId) . '"');
        
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($query);
        
        return $result ?: null;
    }
    
    /**
     * Busca todas las transacciones por ID de factura
     *
     * @param string $invoiceId ID de la factura
     * @return array|null Array de transacciones o null si no existen
     */
    public function findAllByInvoiceId(string $invoiceId): ?array
    {
        $query = new DbQuery();
        $query->select('*')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_invoice = "' . pSQL($invoiceId) . '"');
        
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        
        return $result ?: null;
    }
    
    /**
     * Busca una transacción por carrito, clave de cliente y factura
     *
     * @param int $cartId ID del carrito
     * @param string $customerKey Clave del cliente
     * @param string $invoiceId ID de la factura
     * @return array|null Datos de la transacción o null si no existe
     */
    public function findByCartAndKey(
        int $cartId,
        string $customerKey,
        string $invoiceId
    ): ?array {
        $query = new DbQuery();
        $query->select('*')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_cart = ' . (int) $cartId)
              ->where('id_custom_client = "' . pSQL($customerKey) . '"')
              ->where('id_invoice = "' . pSQL($invoiceId) . '"');
        
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($query);
        
        return $result ?: null;
    }
    
    /**
     * Busca todas las transacciones por carrito, clave de cliente y factura
     *
     * @param int $cartId ID del carrito
     * @param string $customerKey Clave del cliente
     * @param string $invoiceId ID de la factura
     * @return array|null Array de transacciones o null si no existen
     */
    public function findAllByCartAndKey(
        int $cartId,
        string $customerKey,
        string $invoiceId
    ): ?array {
        $query = new DbQuery();
        $query->select('*')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_cart = ' . (int) $cartId)
              ->where('id_custom_client = "' . pSQL($customerKey) . '"')
              ->where('id_invoice = "' . pSQL($invoiceId) . '"');
        
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        
        return $result ?: null;
    }
    
    /**
     * Busca una transacción por ID de pedido
     *
     * @param int $orderId ID del pedido
     * @return array|null Datos de la transacción o null si no existe
     */
    public function findByOrderId(int $orderId): ?array
    {
        $query = new DbQuery();
        $query->select('*')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_order = ' . (int) $orderId);
        
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($query);
        
        return $result ?: null;
    }
    
    /**
     * Actualiza una transacción por ID de factura
     *
     * @param string $invoiceId ID de la factura
     * @param array $data Datos a actualizar
     * @return bool True si la actualización fue exitosa
     */
    public function update(string $invoiceId, array $data): bool
    {
        $updateData = [];
        
        if (isset($data['petition_state'])) {
            $updateData['petition_state'] = pSQL($data['petition_state']);
        }
        
        if (isset($data['state_detail'])) {
            $updateData['state_detail'] = pSQL($data['state_detail']);
        }
        
        if (isset($data['id_order'])) {
            $updateData['id_order'] = (int) $data['id_order'];
        }
        
        if (empty($updateData)) {
            return false;
        }
        
        return Db::getInstance()->update(
            FrakmentaConstants::TABLE_NAME,
            $updateData,
            'id_invoice = "' . pSQL($invoiceId) . '"'
        );
    }
    
    /**
     * Incrementa el contador de intentos de una transacción
     *
     * @param int $cartId ID del carrito
     * @param string $customerKey Clave del cliente
     * @param string $invoiceId ID de la factura
     * @return int Número de intentos actual
     */
    public function incrementAttempts(
        int $cartId,
        string $customerKey,
        string $invoiceId
    ): int {
        Db::getInstance()->execute(
            'UPDATE `' . $this->tableName . '` 
             SET attempts = attempts + 1 
             WHERE id_cart = ' . (int) $cartId . ' 
             AND id_custom_client = "' . pSQL($customerKey) . '" 
             AND id_invoice = "' . pSQL($invoiceId) . '"'
        );
        
        $query = new DbQuery();
        $query->select('attempts')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_cart = ' . (int) $cartId)
              ->where('id_custom_client = "' . pSQL($customerKey) . '"')
              ->where('id_invoice = "' . pSQL($invoiceId) . '"');
        
        return (int) Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);
    }
    
    /**
     * Verifica si un pedido fue pagado con Frakmenta
     *
     * @param int $orderId ID del pedido
     * @return bool True si el pedido fue pagado con Frakmenta
     */
    public function isOrderPaidWithFrakmenta(int $orderId): bool
    {
        $query = new DbQuery();
        $query->select('COUNT(*)')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_order = ' . (int) $orderId);
        
        return (int) Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query) > 0;
    }
    
    /**
     * Obtiene el ID de operación de un pedido
     *
     * @param int $orderId ID del pedido
     * @return int|null ID de operación o null si no existe
     */
    public function getOperationIdByOrder(int $orderId): ?int
    {
        $query = new DbQuery();
        $query->select('id_operation')
              ->from(FrakmentaConstants::TABLE_NAME)
              ->where('id_order = ' . (int) $orderId);
        
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);
        
        return $result ? (int) $result : null;
    }
}
