<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

namespace Frakmenta\Exception;

/**
 * Class FrakmentaApiException
 * 
 * Excepción específica para errores de la API de Frakmenta
 * Se lanza cuando hay problemas de comunicación o respuestas de error de la API
 * 
 * @package Frakmenta\Exception
 */
class FrakmentaApiException extends FrakmentaException
{
    /**
     * @var array|null Datos adicionales de la respuesta de error
     */
    private ?array $responseData = null;
    
    /**
     * @var int|null Código HTTP de la respuesta
     */
    private ?int $httpCode = null;
    
    /**
     * Constructor de FrakmentaApiException
     *
     * @param string $message Mensaje de error
     * @param int $code Código de error
     * @param int|null $httpCode Código HTTP de la respuesta
     * @param array|null $responseData Datos adicionales de la respuesta
     * @param \Exception|null $previous Excepción anterior
     */
    public function __construct(
        string $message = '',
        int $code = 0,
        ?int $httpCode = null,
        ?array $responseData = null,
        ?\Exception $previous = null
    ) {
        parent::__construct($message, $code, $previous);
        $this->httpCode = $httpCode;
        $this->responseData = $responseData;
    }
    
    /**
     * Obtiene el código HTTP de la respuesta
     *
     * @return int|null
     */
    public function getHttpCode(): ?int
    {
        return $this->httpCode;
    }
    
    /**
     * Obtiene los datos adicionales de la respuesta
     *
     * @return array|null
     */
    public function getResponseData(): ?array
    {
        return $this->responseData;
    }
    
    /**
     * Verifica si el error fue un error de cliente (4xx)
     *
     * @return bool
     */
    public function isClientError(): bool
    {
        return $this->httpCode >= 400 && $this->httpCode < 500;
    }
    
    /**
     * Verifica si el error fue un error de servidor (5xx)
     *
     * @return bool
     */
    public function isServerError(): bool
    {
        return $this->httpCode >= 500 && $this->httpCode < 600;
    }
}
