<?php
/*
* 2021 Sistemas findirect
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author Jose Baez
*  @author 2021
*  @author <desarrollo@frakmenta.com>
*  @copyright Sistemas findirect
*
*  @license http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
*/

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

if (!defined('_PS_VERSION_')) {
    exit;
}

class Frakmenta extends PaymentModule
{

    protected $_html = '';
    public $_errors = array();
    public $context;
    public $iso_code;
    public $default_country;
    public $psCheckoutCountry = ['ES'];

    const FRAKMENTA_MODULE_NAME = 'frakmenta';
    const FRAKMENTA_URL = 'https://frakmenta.com';
    const FRAKMENTA_DELEGATION = '1';
    const FRAKMENTA_EXIST_ACCOUNT = 0;

    const FRAKMENTA_TEST_URL = 'https://beta2.frakmenta.com';
    const FRAKMENTA_TEST_PUBLIC_KEY = '91bdde4687b07c34a8af8e23c0f6b1a8dfa1a5c72517fb55116c11741fd98a74';
    const FRAKMENTA_TEST_MERCHANT_ID = '30100';
    const FRAKMENTA_TEST_PRIVATE_KEY = 'eb73e21d88eaedab5cf1d3a6a6b558fe7917204e86120120f0f6cf7ea49ebbdc';

    const FRAKMENTA_TEST_MODE = 0;
    const FRAKMENTA_PRODUCTION_MODE = 1;
    const FRAKMENTA_PRODUCT_OPTION = 0;
    const LOCATION_SIMULATOR_DEFAULT = '.product_attributes';

    const ONLY_PRODUCTS = 1;
    const ONLY_DISCOUNTS = 2;
    const BOTH = 3;
    const BOTH_WITHOUT_SHIPPING = 4;
    const ONLY_SHIPPING = 5;
    const ONLY_WRAPPING = 6;
    const ONLY_PRODUCTS_WITHOUT_SHIPPING = 7;

    public function __construct()
    {
        $this->name = Frakmenta::FRAKMENTA_MODULE_NAME;
        $this->tab = 'payments_gateways';
        $this->version = '3.1.0';
        $this->author = 'Sistemas findirect';
        $this->is_eu_compatible = 1;
        $this->controllers = array('payment', 'notifications', 'success');

        $this->currencies = true;
        $this->currencies_mode = 'radio';

        $this->bootstrap = true;
        parent::__construct();
        $country = new Country(Configuration::get('PS_COUNTRY_DEFAULT'));

        $this->displayName = $this->trans('Frakmenta', [], 'Modules.Frakmenta.Admin');
        $this->description = $this->trans('Este módulo te permite ofrecer a tus clientes cómo pagar de forma fácil y segura con frakmenta', [], 'Modules.Frakmenta.Admin');

        $this->confirmUninstall = $this->l('¿Estás seguro de borrar tú configuración?');
        $this->ps_versions_compliancy = array('min' => '1.7.1.0', 'max' => '9.0.1');
        $this->page = basename(__FILE__, '.php');

    }

    public function install()
    {
        if (!parent::install()
            || !$this->registerHook('paymentReturn')
            || !$this->registerHook('displayHeader')
            || !$this->registerHook('actionAdminControllerSetMedia')
            || !$this->registerHook('validate')
            || !$this->registerHook('displayAdminOrderTop')
            || !$this->registerHook('actionFrontControllerSetMedia')
        ) {
            return false;
        }

        if ((_PS_VERSION_ >= '1.5') && (
                /*!$this->registerHook('displayMobileHeader')
                ||*/ !$this->registerHook('displayOrderConfirmation')
            )) {
            return false;
        }

        if ((_PS_VERSION_ >= '1.7.1.0') && (
                !$this->registerHook('paymentOptions')
                ||!$this->registerHook('displayProductExtraContent')
                ||!$this->registerHook('displayProductAdditionalInfo')
                ||!$this->registerHook('displayAfterProductThumbs')
                ||!$this->registerHook('displayFooterProduct')
            )) {
            return false;
        }

        include_once _PS_MODULE_DIR_.$this->name.'/frakmenta_install.php';
        $fk_install = new FrakmentaInstall();
        $fk_install->createTables();
        $fk_install->updateConfiguration();
        $fk_install->moveTopPayments(1);
        return true;
    }

    public function uninstall()
    {
        include_once _PS_MODULE_DIR_.$this->name.'/frakmenta_install.php';
        $fk_install = new FrakmentaInstall();
        $fk_install->deleteConfiguration();

        return parent::uninstall();
    }

    public function HookDisplayFooterProduct()
    {
        if (Configuration::get('FRAKMENTA_PRODUCT_OPTION')==1) {

            $publicKey = Configuration::get('FRAKMENTA_PUBLIC_KEY');

            $merchantLimits = $this->getMerchantLimits($publicKey);

            $id_product  = Tools::getValue('id_product');
            $price_product = Product::getPriceStatic($id_product);

            if ($merchantLimits->data->min_import<$price_product) {
                Media::addJsDef(array(
                    'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
                    'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
                    'fkApiKey'=> Configuration::get('FRAKMENTA_PUBLIC_KEY'),
                    'productPrice' => intval($price_product*100),
                    'fkPlatform' => 'prestashop',
                    'fkPlatformVersion' => _PS_VERSION_
                ));

                $this->context->controller->registerStylesheet(
                    $this->name.'-widget-ecommerce',
                    Configuration::get('FRAKMENTA_URL').'/css/widget-ecommerce.css',
                    [
                        'media' => 'all',
                        'priority' => 200,
                        'server' => 'remote'
                    ]
                );

                $this->context->controller->registerJavascript(
                    $this->name . '-widgetEcommerce',
                    Configuration::get('FRAKMENTA_URL').'/js/widgetEcommerce.js',
                    array('server' => 'remote', 'priority' => 200, 'position' => 'head'
                    ));

                $this->context->controller->registerJavascript(
                    $this->name . '-fk-simulator-prod', 'modules/' .
                    $this->name . '/views/js/frakmenta_front_products.js',
                    array('server' => 'local', 'priority' => 200, 'position' => 'head')
                );
            }
        }
    }

    protected function getPartialRefund($params)
    {
        $this->context->smarty->assign('trans_frakmenta', $this->getTransacationFrakmentaByOrder($params['id_order']));
        $this->context->smarty->assign('frakmenta_url', Configuration::get('FRAKMENTA_URL'));
        return $this->context->smarty->fetch(_PS_MODULE_DIR_ . $this->name . '/views/templates/hook/partialRefund.tpl');
    }

    public function hookDisplayAdminOrderTop($params)
    {
        if ($this->checkOrderPayByFrakmenta($params['id_order']))
            return $this->getPartialRefund($params);

        return '';
    }

    public function getContent()
    {
        global $cookie;

        if (Configuration::get('PS_SSL_ENABLED') && Configuration::get('PS_SSL_ENABLED_EVERYWHERE') && Tools::usingSecureMode() == false) {
            Tools::redirect('https://'.$_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI']);
        }

        if (empty(Configuration::get('PS_ALLOW_HTML_IFRAME'))) {
            Configuration::updateValue('PS_ALLOW_HTML_IFRAME', 1);
        }

        $output = $this->_postProcess();

        if (version_compare(phpversion(), '5.4', '<')) {
            if (version_compare(_PS_VERSION_, '1.6.1', '>=')) {
                $output = $this->displayWarning($this->l('Su servidor no es compatible con la próxima versión del módulo de Frakmenta. Póngase en contacto con su empresa de alojamiento para actualizar la versión de PHP al menos a la versión 5.4 o la más reciente.'));
            } else {
                $output = $this->displayError($this->l('Su servidor no es compatible con la próxima versión del módulo de Frakmenta. Póngase en contacto con su empresa de alojamiento para actualizar la versión de PHP al menos a la versión 5.4 o la más reciente.'));
            }
        }

        $iso_code = Language::getIsoById( (int)$cookie->id_lang);
        $languange_store = $iso_code=='es' ? true : false;

        $languange_store = true;

        // Obtener país de la tienda
        $defaultCountryId = Configuration::get('PS_COUNTRY_DEFAULT');
        $country = new Country($defaultCountryId);
        $countryName = isset($country->name[$cookie->id_lang]) ? $country->name[$cookie->id_lang] : $country->name[1];
        $countryIso = $country->iso_code;

        $this->context->smarty->assign(array(
            'Frakmenta_account' => Configuration::get('FRAKMENTA_ACCOUNT'),
            'Frakmenta_public_key' => Configuration::get('FRAKMENTA_PUBLIC_KEY'),
            'Frakmenta_private_key' => Configuration::get('FRAKMENTA_PRIVATE_KEY'),
            'Frakmenta_merchant_id' => Configuration::get('FRAKMENTA_MERCHANT_ID'),
            'Frakmenta_mode' => Configuration::get('FRAKMENTA_MODE'),
            'Frakmenta_endpoint' => Configuration::get('FRAKMENTA_URL'),
            'Frakmenta_simulator_products' => Configuration::get('FRAKMENTA_PRODUCT_OPTION'),
            'moduleDir' => _MODULE_DIR_ . $this->name,
            'language_store' => $languange_store,
            'currentCountry' => $countryName,
            'currentCountryIso' => $countryIso,
            'Frakmenta_location_simulator' => Configuration::get('FRAKMENTA_LOCATION_SIMULATOR')
        ));

        $output .= $this->fetchTemplate('/views/templates/admin/back_office.tpl');

        return $output;
    }

    /**
     * Renderiza una plantilla del módulo
     * Compatible con PrestaShop 1.7+
     *
     * @param string $name Nombre del archivo de plantilla
     * @return string HTML renderizado
     */
    public function fetchTemplate(string $name): string
    {
        return $this->display(__FILE__, $name);
    }

    /**
     * Hooks methods
     */
    private function getNumberPaymentFrakmenta(){
        $position = -1;
        foreach (PaymentModule::getInstalledPaymentModules() as $payment) {
            $module = Module::getInstanceByName($payment['name']);
            if ($module->active)
                $position++;

            if ($payment['name']=='frakmenta'){
                $position = $position;
                break;
            }
        }
        return $position;
    }

    public function hookActionFrontControllerSetMedia(){
        $this->context->controller->registerStylesheet($this->name . '-frakmenta', 'modules/' . $this->name . '/views/css/frakmenta.css');
        $this->context->controller->registerJavascript($this->name . '-operation-frakmenta', 'modules/' . $this->name . '/views/js/frakmenta_front_operation.js', []);
        $this->context->controller->registerJavascript($this->name . '-validations-frakmenta', 'modules/' . $this->name . '/views/js/frakmenta_front_validations.js', ['position' => 'head']);
        $this->context->controller->registerStylesheet($this->name . '-frakmenta-iframe', 'modules/' . $this->name . '/views/css/iframe-ecommerce.min.css');

        $this->context->controller->registerStylesheet(
            $this->name.'-widget-ecommerce',
            Configuration::get('FRAKMENTA_URL').'/css/widget-ecommerce.css',
            [
                'media' => 'all',
                'priority' => 200,
                'server' => 'remote'
            ]
        );

        $this->context->controller->registerJavascript(
            $this->name . '-widgetEcommerce',
            Configuration::get('FRAKMENTA_URL').'/js/widgetEcommerce.js',
            array('server' => 'remote', 'priority' => 200, 'position' => 'head'
            ));

        Media::addJsDef(array(
            'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiKey'=> Configuration::get('FRAKMENTA_PUBLIC_KEY'),
            'FrakmentaUrlGetToken' => Context::getContext()->link->getModuleLink('frakmenta', 'payment', array('ajax' =>true)),
            'fkPlatform' => 'prestashop',
            'fkPlatformVersion' => _PS_VERSION_
        ));

        if (isset($this->context->cart) && $this->context->cart->id) {
            $this->context->smarty->assign('id_cart', (int) $this->context->cart->id);
        }

        $smarty = $this->context->smarty;
        $smarty->assign(array(
            'ssl_enabled' => Configuration::get('PS_SSL_ENABLED')
        ));
    }

    public function hookPaymentReturn()
    {
        $module = Module::getInstanceByName(FRAKMENTA::FRAKMENTA_MODULE_NAME);

        $frakmenta_transaction = $this->getDbFrakmentaTransactionDetailsByKeyCart($_GET['key'], $_GET['id_cart'], $_GET['invoice']);

        if ($module->id == $_GET['id_module'] && strtolower($frakmenta_transaction[0]['petition_state'])=='ok')
        {
            $this->context->smarty->assign(array(
                'frakmenta_operation' => $frakmenta_transaction[0]['id_operation']
            ));

            return $this->fetchTemplate('payment_return.tpl');
        }
    }

    public function hookDisplayHeader()
    {
        $returnContent = '';
        $this->context->controller->registerStylesheet($this->name . '-frakmenta', 'modules/' . $this->name . '/views/css/frakmenta.css');
        $this->context->controller->registerJavascript($this->name . '-operation-frakmenta', 'modules/' . $this->name . '/views/js/frakmenta_front_operation.js', []);
        $this->context->controller->registerJavascript($this->name . '-validations-frakmenta', 'modules/' . $this->name . '/views/js/frakmenta_front_validations.js', ['position' => 'head']);
        $this->context->controller->registerStylesheet($this->name . '-frakmenta-iframe', 'modules/' . $this->name . '/views/css/iframe-ecommerce.min.css');

        $this->context->controller->registerStylesheet(
            $this->name.'-widget-ecommerce',
            Configuration::get('FRAKMENTA_URL').'/css/widget-ecommerce.css',
            [
                'media' => 'all',
                'priority' => 200,
                'server' => 'remote'
            ]
        );

        $this->context->controller->registerJavascript(
            $this->name . '-widgetEcommerce',
            Configuration::get('FRAKMENTA_URL').'/js/widgetEcommerce.js',
            array('server' => 'remote', 'priority' => 200, 'position' => 'head'
            ));

        Media::addJsDef(array(
            'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
            'fkApiKey'=> Configuration::get('FRAKMENTA_PUBLIC_KEY'),
            'FrakmentaUrlGetToken' => Context::getContext()->link->getModuleLink('frakmenta', 'payment', array('ajax' =>true)),
            'fkPlatform' => 'prestashop',
            'fkPlatformVersion' => _PS_VERSION_
        ));


        if (Configuration::get('FRAKMENTA_PRODUCT_OPTION')==1) {

            Media::addJsDef(array(
                'fkLocationSimulador' => empty(Configuration::get('FRAKMENTA_LOCATION_SIMULATOR'))? '.product_attributes' : Configuration::get('FRAKMENTA_LOCATION_SIMULATOR'),
                'logoFrakmenta' => $this->context->shop->getBaseURI().'modules/'.$this->name.'/frakmenta-logo-original.png'
            ));

            $this->context->controller->registerJavascript(
                $this->name . '-fk-simulator-prod', 'modules/' .
                $this->name . '/views/js/frakmenta_front_products.js',
                array('server' => 'local', 'priority' => 200, 'position' => 'bottom')
            );

        }

        if (isset($this->context->cart) && $this->context->cart->id) {
            $this->context->smarty->assign([
                    'id_cart' => (int) $this->context->cart->id,
                    'shop_name' => Configuration::get('PS_SHOP_NAME')
                ]
            );
        }

        $smarty = $this->context->smarty;
        $smarty->assign(array(
            'ssl_enabled' => Configuration::get('PS_SSL_ENABLED'),
            'shop_name' => Configuration::get('PS_SHOP_NAME')
        ));

        $returnContent .= $this->context->smarty->fetch('module:frakmenta/views/templates/front/prefetch.tpl');

        return $returnContent;
    }

    private function isAMPInstall(){
        $mod = false;
        foreach (Module::getModulesInstalled() as $modules) {
            if (strpos(strtoupper($modules['name']),'AMP')!==FALSE && $modules['active']==1){
                $mod = true;
                break;
            }
        }
        return $mod;
    }



    private function getMerchantLimits($publicKey)
    {
        $dataMerchant = $this->limitsMerchantByPublicKey($publicKey);

        // Validar si la respuesta es válida y tiene datos
        if (!is_object($dataMerchant) || 
            !isset($dataMerchant->status) || 
            $dataMerchant->status != 'ok' || 
            !isset($dataMerchant->data) || 
            $dataMerchant->data == null)
        {
            // Crear objeto con valores por defecto si no hay respuesta válida
            $dataMerchant = new stdClass();
            $dataMerchant->status = 'ok';
            $dataMerchant->data = new stdClass();
            $dataMerchant->data->max_import = 1000;
            $dataMerchant->data->min_import = 59;
        }

        return $dataMerchant;
    }

    private function frakmenta_api_connection($url, $data, $method){

        $curl = curl_init();

        $params = array(
            'Content-Type: application/json'
        );

        if ($method=='POST')
            array_push($params, 'Content-Length: ' . strlen($data));

        $request = [
            CURLOPT_URL => ($method=='GET' ? $url.http_build_query($data) : $url),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => $params
        ];

        if ($method=='POST')
            $request[CURLOPT_POSTFIELDS] = $data;

        curl_setopt_array($curl, $request);

        $response = curl_exec($curl);

        curl_close($curl);

        return utf8_encode($response);
    }

    private function limitsMerchantByPublicKey($publicKey){
        $url = Configuration::get('FRAKMENTA_URL').'/api/fk/v2/limits?';

        $data = [
            "apikey" => $publicKey
        ];

        $response = $this->frakmenta_api_connection($url, $data, 'GET');

        return json_decode($response);
    }


    public function hookPaymentOptions($params)
    {

        if (!$this->active) {
            return;
        }
        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $publicKey = Configuration::get('FRAKMENTA_PUBLIC_KEY');

        $merchantLimits = $this->getMerchantLimits($publicKey);

        $cart = $this->context->cart;
        $totalOrder = (float)($cart->getOrderTotal(true, Cart::BOTH));
        $pendingOrderFk = $merchantLimits->data->min_import - $totalOrder;

        if (intval($merchantLimits->data->max_import) < $totalOrder)
            return;

        $minAmountDiff = $merchantLimits->data->min_import - $totalOrder;
        
        // LOG: Información de depuración
        PrestaShopLogger::addLog(
            'Frakmenta hookPaymentOptions DEBUG: ' .
            'totalOrder=' . $totalOrder . 
            ', min_import=' . $merchantLimits->data->min_import . 
            ', pendingOrderFk=' . $pendingOrderFk .
            ', minAmountDiff=' . $minAmountDiff . 
            ', formatted=' . number_format($minAmountDiff, 2, ',', ''),
            1,
            null,
            'Frakmenta',
            null,
            true
        );
        
        $variables = [
            'Frakmenta_minAmount' => (float) number_format($pendingOrderFk, 2, ',', ''),
            'Frakmenta_price_order' => $totalOrder*100,
            'Frakmenta_public_key' => $publicKey,
            'Frakmenta_merchant_id' => Configuration::get('FRAKMENTA_MERCHANT_ID'),
            'Frakmenta_endpoint' => Configuration::get('FRAKMENTA_URL'),
            'baseURI' => $this->context->shop->getBaseURI(),
            'Frakmenta_minAmount' => number_format($minAmountDiff, 2, ',', ''),
            'FrakmentaKO' =>  empty(Tools::getValue('fk')) ? 'OK' : trim(Tools::getValue('fk'))
        ];

        $titleFrakmentaPay = "Paga a plazos con frakmenta";

        if ($pendingOrderFk>0)
            $titleFrakmentaPay = "Te faltan $pendingOrderFk € para pagar a plazos con frakmenta";

        //Para sitios con modulos que sobre-escriben el tema y no llaman los hooks genericos
        if ($this->isAMPInstall()){

            $base = _PS_BASE_URL_;
            if (Configuration::get('PS_SSL_ENABLED')==1)
                $base = str_replace('http', 'https', _PS_BASE_URL_);

            $baseUri = Context::getContext()->shop->getBaseURI();

            $resources[] = 'https://static.frakmenta.com/js/jquery.min.js';
            $resources[] = Configuration::get('FRAKMENTA_URL').'/js/widgetEcommerce.js'. '?v=' . $this->version;
            $resources[] = $base.$baseUri.'modules/' . $this->name . '/views/js/frakmenta_front_operation.js'. '?v=' . $this->version;
            $resources[] = $base.$baseUri.'modules/' . $this->name . '/views/js/frakmenta_front_validations.js'. '?v=' . $this->version;

            $styles[] = $base.$baseUri.'modules/' . $this->name . '/views/css/frakmenta.css'. '?v=' . $this->version;
            $styles[] = $base.$baseUri.'modules/' . $this->name . '/views/css/iframe-ecommerce.min.css'. '?v=' . $this->version;
            $styles[] = Configuration::get('FRAKMENTA_URL').'/css/widget-ecommerce.css'. '?v=' . $this->version;

            $this->context->smarty->assign('resources', $resources);
            $this->context->smarty->assign('styles', $styles);

            $variables['fkEcommerceUrl'] = Configuration::get('FRAKMENTA_URL');
            $variables['fkApiUrl'] = Configuration::get('FRAKMENTA_URL');
            $variables['fkApiKey'] = Configuration::get('FRAKMENTA_PUBLIC_KEY');
            $variables['FrakmentaUrlGetToken'] = Context::getContext()->link->getModuleLink('frakmenta', 'payment', array('ajax' =>true));
            $variables['FrakmentaPosition'] = $this->getNumberPaymentFrakmenta();
        }

        $this->smarty->assign($variables);

        $newOption = new PaymentOption();
        $newOption->setCallToActionText($titleFrakmentaPay);
        $newOption->setModuleName($this->name);
        //$newOption->setLogo(Media::getMediaPath(_PS_MODULE_DIR_ . $this->name . '/logo_card.png')); //11/12/2021 Disable logo
        $newOption->setAdditionalInformation($this->fetch('module:frakmenta/views/templates/front/info_payment.tpl'));

        return [$newOption];

    }

    public function hookDisplayProductAdditionalInfo($params) {
        if (Configuration::get('FRAKMENTA_PRODUCT_OPTION')==1) {

            $publicKey = Configuration::get('FRAKMENTA_PUBLIC_KEY');

            $merchantLimits = $this->getMerchantLimits($publicKey);

            $id_product  = Tools::getValue('id_product');
            $price_product = Product::getPriceStatic($id_product);

            if ($merchantLimits->data->min_import<$price_product) {
                Media::addJsDef(array(
                    'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
                    'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
                    'fkApiKey'=> Configuration::get('FRAKMENTA_PUBLIC_KEY'),
                    'productPrice' => intval($price_product*100),
                    'fkPlatform' => 'prestashop',
                    'fkPlatformVersion' => _PS_VERSION_
                ));

                $this->context->controller->registerStylesheet(
                    $this->name.'-widget-ecommerce',
                    Configuration::get('FRAKMENTA_URL').'/css/widget-ecommerce.css',
                    [
                        'media' => 'all',
                        'priority' => 200,
                        'server' => 'remote'
                    ]
                );

                $this->context->controller->registerJavascript(
                    $this->name . '-widgetEcommerce',
                    Configuration::get('FRAKMENTA_URL').'/js/widgetEcommerce.js',
                    array('server' => 'remote', 'priority' => 200, 'position' => 'head'
                    ));

                $this->context->controller->registerJavascript(
                    $this->name . '-fk-simulator-prod', 'modules/' .
                    $this->name . '/views/js/frakmenta_front_products.js',
                    array('server' => 'local', 'priority' => 200, 'position' => 'bottom')
                );
            }
        }
    }

    public function hookDisplayAfterProductThumbs(){
        // die('frakmenta2');
        null;
    }

    public function hookDisplayProductExtraContent($params) {

        if (Configuration::get('FRAKMENTA_PRODUCT_OPTION')==1) {

            $publicKey = Configuration::get('FRAKMENTA_PUBLIC_KEY');

            $merchantLimits = $this->getMerchantLimits($publicKey);

            $id_product  = Tools::getValue('id_product');
            $price_product = Product::getPriceStatic($id_product);

            if ($merchantLimits->data->min_import<$price_product) {
                Media::addJsDef(array(
                    'fkEcommerceUrl' => Configuration::get('FRAKMENTA_URL'),
                    'fkApiUrl' => Configuration::get('FRAKMENTA_URL'),
                    'fkApiKey'=> Configuration::get('FRAKMENTA_PUBLIC_KEY'),
                    'productPrice' => intval($price_product*100),
                    'fkPlatform' => 'prestashop',
                    'fkPlatformVersion' => _PS_VERSION_
                ));

                $this->context->controller->registerStylesheet(
                    $this->name.'-widget-ecommerce',
                    Configuration::get('FRAKMENTA_URL').'/css/widget-ecommerce.css',
                    [
                        'media' => 'all',
                        'priority' => 200,
                        'server' => 'remote'
                    ]
                );

                $this->context->controller->registerJavascript(
                    $this->name . '-widgetEcommerce',
                    Configuration::get('FRAKMENTA_URL').'/js/widgetEcommerce.js',
                    array('server' => 'remote', 'priority' => 200, 'position' => 'head'
                    ));

                $this->context->controller->registerJavascript(
                    $this->name . '-fk-simulator-prod', 'modules/' .
                    $this->name . '/views/js/frakmenta_front_products.js',
                    array('server' => 'local', 'priority' => 200, 'position' => 'bottom')
                );
            }
        }
    }



    /**
     * Hook para cargar assets en el backoffice
     * Compatible con PrestaShop 1.7+
     */
        public function hookActionAdminControllerSetMedia()
    {
        if ((strcmp(Tools::getValue('configure'), $this->name) === 0) ||
            (strcmp(Tools::getValue('module_name'), $this->name) === 0)) {
            header('Clear-Site-Data: "cache"');

            $merchant_id = Configuration::get('FRAKMENTA_MERCHANT_ID');
            $delegation = '1';
            $private_key = Configuration::get('FRAKMENTA_PRIVATE_KEY');
            $public_key = Configuration::get('FRAKMENTA_PUBLIC_KEY');
            $signature = hash('sha256', $merchant_id . '|' . $delegation . '|' . $private_key);

            Media::addJsDef([
                'frakmenta_merchant_id_validation' =>  $merchant_id,
                'frakmenta_public_key_validation' =>  $public_key,
                'frakmenta_sign_validation' =>  $signature,
            ]);

            // Rutas de assets
            $cssPath = $this->_path . 'views/css/frakmenta.css';
            $jsPath = $this->_path . 'views/js/frakmenta_back_office.js';

            // Registrar CSS (compatibilidad con distintas versiones / proxies)
            if (method_exists($this->context->controller, 'registerStylesheet')) {
                $this->context->controller->registerStylesheet(
                    'module-frakmenta-admin',
                    $cssPath,
                    [
                        'media' => 'all',
                        'priority' => 150,
                    ]
                );
            } elseif (method_exists($this->context->controller, 'addCSS')) {
                $this->context->controller->addCSS($cssPath, 'all');
            } else {
                // Fallback: intentar usar addCSS si existe en la clase Media (antiguo)
                if (method_exists('Media', 'addCss')) {
                    Media::addCss($cssPath);
                }
            }
        }

        // Registrar JavaScript (compatibilidad con distintas versiones / proxies)
        $jsPathGlobal = $this->_path . 'views/js/frakmenta_back_office.js';
        if (method_exists($this->context->controller, 'registerJavascript')) {
            $this->context->controller->registerJavascript(
                'module-frakmenta-admin-js',
                $jsPathGlobal,
                ['position' => 'bottom', 'priority' => 150]
            );
        } elseif (method_exists($this->context->controller, 'addJS')) {
            $this->context->controller->addJS($jsPathGlobal);
        } else {
            // último recurso: insertar con Media si existe
            if (method_exists('Media', 'addJs')) {
                Media::addJs($jsPathGlobal);
            }
        }

    }

    public function hookDisplayOrderConfirmation()
    {
        null;
    }

     // Añadido: método vacío para el hook 'validate' registrado en install()
    public function hookValidate($params)
    {
        // Manejo opcional de la validación. Actualmente no hace nada,
        return;
    }

    public function getPaymentMethods()
    {
        $country = new Country((int) Configuration::get('PS_COUNTRY_DEFAULT'));
        return AuthenticatePaymentMethods::authenticatePaymentMethodByCountry($country->iso_code);
    }

    public function getCountryCode()
    {
        $cart = new Cart((int) $this->context->cookie->id_cart);
        $address = new Address((int) $cart->id_address_invoice);
        $country = new Country((int) $address->id_country);

        return $country->iso_code;
    }



    private function _postProcess()
    {
        if (Tools::isSubmit('submitButton')) {
            if (Tools::getValue('fk_mode') == 0)
            {
                Configuration::updateValue('FRAKMENTA_ACCOUNT', trim(Tools::getValue('fk_account')));
                Configuration::updateValue('FRAKMENTA_URL', Frakmenta::FRAKMENTA_TEST_URL);
                Configuration::updateValue('FRAKMENTA_PUBLIC_KEY', Frakmenta::FRAKMENTA_TEST_PUBLIC_KEY);
                Configuration::updateValue('FRAKMENTA_PRIVATE_KEY', Frakmenta::FRAKMENTA_TEST_PRIVATE_KEY);
                Configuration::updateValue('FRAKMENTA_MERCHANT_ID', Frakmenta::FRAKMENTA_TEST_MERCHANT_ID);
                Configuration::updateValue('FRAKMENTA_MODE', Frakmenta::FRAKMENTA_TEST_MODE);
                Configuration::updateValue('FRAKMENTA_PRODUCT_OPTION', trim(Tools::getValue('fk_sim_product')));
                Configuration::updateValue('FRAKMENTA_LOCATION_SIMULATOR', trim(Tools::getValue('fk_location_simulator')));
            }
            else
            {
                Configuration::updateValue('FRAKMENTA_URL', Frakmenta::FRAKMENTA_URL);
                Configuration::updateValue('FRAKMENTA_PUBLIC_KEY', trim(Tools::getValue('fk_public_key')));
                Configuration::updateValue('FRAKMENTA_PRIVATE_KEY', trim(Tools::getValue('fk_private_key')));
                Configuration::updateValue('FRAKMENTA_MERCHANT_ID', trim(Tools::getValue('fk_merchant_id')));
                Configuration::updateValue('FRAKMENTA_MODE', Frakmenta::FRAKMENTA_PRODUCTION_MODE);
                Configuration::updateValue('FRAKMENTA_ACCOUNT', trim(Tools::getValue('fk_account')));
                Configuration::updateValue('FRAKMENTA_PRODUCT_OPTION', trim(Tools::getValue('fk_sim_product')));
                Configuration::updateValue('FRAKMENTA_LOCATION_SIMULATOR', trim(Tools::getValue('fk_location_simulator')));
            }

            Configuration::updateValue('FRAKMENTA_DELEGATION', Frakmenta::FRAKMENTA_DELEGATION);
            $this->context->smarty->assign('frakmenta_save_sucess', true);
        } else {
            $this->_html = $this->displayError(implode('<br />', $this->_errors)); // Not displayed at this time
            $this->context->smarty->assign('Frakmenta_save_failure', true);
        }

        return $this->loadLangDefault();
    }

    private function loadLangDefault()
    {
        $this->default_country = (int) Configuration::get('PS_COUNTRY_DEFAULT');
        $country = new Country($this->default_country);
        $this->iso_code = Tools::strtoupper($country->iso_code);
    }

    private function checkCurrency($cart)
    {
        $currency_module = $this->getCurrency((int) $cart->id_currency);

        if ((int) $cart->id_currency == (int) $currency_module->id) {
            return true;
        } else {
            return false;
        }
    }

    public static function getShopDomainSsl($http = false, $entities = false)
    {
        if (method_exists('Tools', 'getShopDomainSsl')) {
            return Tools::getShopDomainSsl($http, $entities);
        } else {
            if (!($domain = Configuration::get('PS_SHOP_DOMAIN_SSL'))) {
                $domain = self::getHttpHost();
            }

            if ($entities) {
                $domain = htmlspecialchars($domain, ENT_COMPAT, 'UTF-8');
            }

            if ($http) {
                $domain = (Configuration::get('PS_SSL_ENABLED') ? 'https://' : 'http://').$domain;
            }

            return $domain;
        }
    }

    /**
     * Check if the current page use SSL connection on not
     *
     * @return bool uses SSL
     */
    public function usingSecureMode()
    {
        if (isset($_SERVER['HTTPS'])) {
            return ($_SERVER['HTTPS'] == 1 || Tools::strtolower($_SERVER['HTTPS']) == 'on');
        }

        // $_SERVER['SSL'] exists only in some specific configuration
        if (isset($_SERVER['SSL'])) {
            return ($_SERVER['SSL'] == 1 || Tools::strtolower($_SERVER['SSL']) == 'on');
        }

        return false;
    }

    public function getCurrentUrl()
    {
        $protocol_link = $this->usingSecureMode() ? 'https://' : 'http://';
        $request = $_SERVER['REQUEST_URI'];
        $pos = strpos($request, '?');

        if (($pos !== false) && ($pos >= 0)) {
            $request = Tools::substr($request, 0, $pos);
        }

        $params = urlencode($_SERVER['QUERY_STRING']);
        if ($params) {
            $url = $protocol_link.Tools::getShopDomainSsl().$request.'?'.$params;
        } else {
            $url = $protocol_link.Tools::getShopDomainSsl().$request;
        }

        return $url;
    }

    public function createDbFrakmentaTransaction($id_cart, $id_custom_client, $responseOperation, $id_invoice, $ammount)
    {
        $responseOperation = json_decode($responseOperation);
        $petition_state = (strtoupper($responseOperation->status)=='OK') ? 'initialized' : 'failed';
        $id_operation = (strtoupper($responseOperation->status)=='OK') ? $responseOperation->data->operation_id : -1;

        return Db::getInstance(_PS_USE_SQL_SLAVE_)->execute('INSERT INTO `'._DB_PREFIX_.'frakmenta_petitions` (`id_cart`, `id_custom_client`, `id_operation`, `id_invoice`, `ammount`, `petition_state`, `attempts`) values ('.$id_cart.', "'.$id_custom_client.'", '.$id_operation.', "'.$id_invoice.'", '.$ammount.', "'.$petition_state.'", 0)');
    }

    public function getDbFrakmentaTransactionDetailsByInvoiceId($invoice_id)
    {
        $resultDB = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('SELECT * FROM `'._DB_PREFIX_.'frakmenta_petitions` WHERE id_invoice="'.$invoice_id.'"');

        if (count($resultDB)>0)
            return $resultDB;

        return false;
    }

    public function getDbFrakmentaTransactionDetailsByKeyCart($client_key, $id_cart, $invoice_id)
    {
        $resultDB = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('SELECT * FROM `'._DB_PREFIX_.'frakmenta_petitions` WHERE id_custom_client="'.$client_key.'" AND id_cart='.$id_cart.' AND id_invoice="'.$invoice_id.'"');

        if (count($resultDB)>0)
            return $resultDB;

        return false;
    }

    public function updateAttemptgetStatusTransactionFk($client_key, $id_cart, $invoice_id)
    {
        Db::getInstance(_PS_USE_SQL_SLAVE_)->execute('UPDATE `'._DB_PREFIX_.'frakmenta_petitions` SET attempts = attempts + 1 WHERE id_custom_client="'.$client_key.'" AND id_cart='.$id_cart.' AND id_invoice="'.$invoice_id.'"' );

        return Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('SELECT attempts FROM `'._DB_PREFIX_.'frakmenta_petitions` WHERE id_custom_client="'.$client_key.'" AND id_cart='.$id_cart.' AND id_invoice="'.$invoice_id.'"');
    }

    public function updateDbFrakmentaTransactionDetails($invoice_id, $estado, $detalleEstado, $order_id)
    {
        return Db::getInstance(_PS_USE_SQL_SLAVE_)->execute('UPDATE `'._DB_PREFIX_.'frakmenta_petitions` SET petition_state="'.$estado.'", state_detail="'.$detalleEstado.'", id_order='.$order_id.', update_date=CURRENT_TIMESTAMP() WHERE id_invoice="'.$invoice_id.'"');
    }

    public function frakmentaManagerTransaction($detalleTransaccionBD, $resultTransaccion)
    {
        if (count($detalleTransaccionBD)>0 && strtoupper($resultTransaccion)=='OK')
        {

            $frakmenta = Module::getInstanceByName(Frakmenta::FRAKMENTA_MODULE_NAME);

            $cart = Context::getContext()->cart = new Cart((int) $detalleTransaccionBD[0]['id_cart']);

            $address = new Address((int) Context::getContext()->cart->id_address_invoice);
            $country = Context::getContext()->country = new Country((int) $address->id_country);
            $customer = Context::getContext()->customer = new Customer((int) Context::getContext()->cart->id_customer);
            $language = Context::getContext()->language = new Language((int) Context::getContext()->cart->id_lang);
            $currency = Context::getContext()->currency = new Currency((int) Context::getContext()->cart->id_currency);

            if (isset(Context::getContext()->cart->id_shop)) {
                Context::getContext()->shop = new Shop(Context::getContext()->cart->id_shop);
            }

            if ($cart->id_customer == 0 OR $cart->id_address_delivery == 0 OR $cart->id_address_invoice == 0)
                return false;

            $customer = new Customer((int)$cart->id_customer);
            $total = (float)($cart->getOrderTotal(true, Cart::BOTH));

            $frakmenta->validateOrder($cart->id, Configuration::get('PS_OS_PAYMENT'), $total, $frakmenta->displayName, NULL, array(), (int)$currency->id, false, $customer->secure_key);

            $order = new Order($frakmenta->currentOrder);

            if ($order)
            {
                $history = new OrderHistory();
                $history->id_order = (int) $order->id;
                $history->changeIdOrderState(2, $history->id_order);
                $history->addWithemail();
                $history->save();

                Db::getInstance()->update('order_payment', array(
                    'transaction_id' => pSQL($detalleTransaccionBD[0]['id_operation']),
                ), 'order_reference = "'.pSQL($order->reference).'"');

                return $order->id;
            }
        }
        return 0;
    }

    public function requestFrakmentaOperation($url, $private_key, $merchant_id, $cart, $address, $country, $ps_customer)
    {
        $products = $cart->getProducts(true);
        $fecha_customer = date_create($ps_customer->date_add);
        $invoice_id = hash("sha256", (strval($merchant_id.'-'.$cart->id.'-'.date('YmdHis'))));
        $product_price = $cart->getOrderTotal(true, Cart::BOTH) * 100;
        $cart_products = [];

        foreach ($products as $products) {
            $cart_products[] = [
                'id' => $products['id_product'],
                'name' => substr($products['name'], 0, 900),
                'quantity' => $products['cart_quantity'],
                'price' => number_format($products['price'], 2, '.', ''),
                'tax_rate' => $products['rate'],
                'description' => strip_tags(substr($products['description_short'], 0, 890)),
                'url' => 'https://frakmenta.com',
                'image_url' => 'https://frakmenta.com'
            ];
        }

        $customer = [
            'identification' => [
                'nif' => $address->dni,
                'legal_first_name' => empty($address->firstname) ? $ps_customer->firstname : $address->firstname,
                'legal_last_name' => empty($address->lastname) ? $ps_customer->lastname : $address->lastname,
                'date_of_birth' => $ps_customer->birthday,
                'mobile_phone_number' => empty($address->phone_mobile) ? $address->phone : $address->phone_mobile,
                'email' => $ps_customer->email
            ],
            'address' => [
                'line_1' => $address->address1,
                'line_2' => empty($address->address2) ? " " : $address->address2,
                'phone' => $address->phone,
                'city' => $address->city,
                'state' => $address->city,
                'county' => $country->name[1],
                'country_code' => $country->iso_code,
                'postcode' => $address->postcode,
            ],
            "store_details" => [
                "customer_date_joined" => $fecha_customer->format('Y-m-d'),
                "customer_last_login" => $fecha_customer->format('Y-m-d')
            ],
            "financial" => [
                "salary" => 0,
                "currency" => "EUR",
                "employment_status" => "N/A",
                "contract_type" => "N/A"
            ],
            'other_data' => [
                ['name' => 'Tienda', 'type' => 'STRING', 'value' => Configuration::get('PS_SHOP_NAME')],
                ['name' => 'Ecommerce', 'type' => 'STRING', 'value' => 'PRESTASHOP'],
                ['name' => 'Version', 'type' => 'STRING', 'value' => _PS_VERSION_],
                ['name' => 'Enviroment', 'type' => 'STRING', 'value' => Configuration::get('FRAKMENTA_MODE')==0?'TEST':'PRODUCTION']
            ]
        ];

        $order = [
            'id' => strval($cart->id),
            'products' => $cart_products
        ];

        $success_url = Context::getContext()->link->getModuleLink('frakmenta', 'success').'?id='.$ps_customer->secure_key.'&cart='.$cart->id.'&invoice='.$invoice_id;
        $notification_url = Context::getContext()->link->getModuleLink('frakmenta', 'notifications');

        if (_PS_BASE_URL_ == 'http' && Configuration::get('FRAKMENTA_MODE') == 0) {
            $success_url = str_replace('https', 'http', $success_url);
            $notification_url = str_replace('https', 'http', $notification_url);
        }

        $flowConfig = [
            'success_url' => $success_url,
            'notification_url' => $notification_url,
            'ko_url' => _PS_BASE_URL_.Context::getContext()->shop->getBaseURI().'index.php?controller=order&step=3&fk=KO',
        ];

        $data = [
            'merchant_id' => $merchant_id,
            'invoice_id' => $invoice_id,
            'product_price' => $product_price,
            'currency_code' => 'EUR',
            'delegation' => '1',
            'type' => 'e-commerce',
            'customer' => $customer,
            'order' => $order,
            'flow_config' => $flowConfig,
            'other_data' => [['name' => 'N/A', 'type' => 'STRING', 'value' => 'N/A']]
        ];

        $signature = hash("sha256",
            $data["merchant_id"].'|'.
            $data["delegation"].'|'.
            $data["type"].'|'.
            $data["invoice_id"].'|'.
            $data["product_price"].'|'.
            $data["currency_code"].'|'.
            $private_key,
            FALSE
        );

        $data['signature'] = $signature;

        $errors = [];

        $data_encoded = json_encode($data);

        if (json_last_error()) {
            $errors[] = sprintf('Error en json_encode (%s)', json_last_error_msg());
        }

        $url .= "/operations";

        $response = $this->frakmenta_api_connection($url, $data_encoded, 'POST');
        $response_frakmenta = json_decode($response);

        if (json_last_error()) {
            $errors[] = sprintf('Error en json_decode (%s)', json_last_error_msg());
        }

        if (strtoupper($response_frakmenta->status)!='OK'){
            $this->reportErrorFrakmenta($data, $response, $errors);
        }

        $this->createDbFrakmentaTransaction($cart->id, $ps_customer->secure_key, $response, $invoice_id, $product_price);

        return $response_frakmenta;
    }

    private function checkOrderPayByFrakmenta($order)
    {
        $resultDB = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('SELECT * FROM `'._DB_PREFIX_.'frakmenta_petitions` WHERE id_order ='.$order);

        if (count($resultDB)>0)
            return true;

        return false;
    }

    private function getTransacationFrakmentaByOrder($order)
    {
        $resultDB = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('SELECT id_operation FROM `'._DB_PREFIX_.'frakmenta_petitions` WHERE id_order ='.$order);

        return $resultDB[0]['id_operation'];
    }

    private function reportErrorFrakmenta($request_data, $response_data, $error_messages)
    {
        $conexion = Configuration::get('FRAKMENTA_MODE') == 0 ? "Pruebas" : "Producción";

        $msj = '<b>Información del comercio</b> <br><br>';
        $msj .= '<b>Tienda:</b> '.Configuration::get('PS_SHOP_NAME').'<br>';
        $msj .= '<b>Conexión:</b> '.$conexion.'<br>';

        if (!empty($error_messages)) {
            $msj .= '<b>Errores en plugin:</b> ';
            foreach ($error_messages as $error_index => $error_message) {
                $msj .= sprintf('<pre>(%s) %s</pre>', $error_index + 1, $error_message);
            }
            $msj .= '<br />';
        }

        $msj .= '<b>Respuesta API:</b> <pre>'.print_r($response_data, true).'</pre>';
        $msj .= '<b>Request:</b> <pre>'.print_r($request_data, true).'</pre>';

        $this->sendEmail($msj);
    }

    public function getStatusOperationFrakmenta($idOperation){
        $signature = hash("sha256",
            Configuration::get('FRAKMENTA_MERCHANT_ID').'|'.
            '1|'.
            'e-commerce'.'|'.
            $idOperation.'|'.
            Configuration::get('FRAKMENTA_PRIVATE_KEY'),
            FALSE);

        $data = json_encode([
            "merchant_id" => Configuration::get('FRAKMENTA_MERCHANT_ID'),
            "delegation" => 1,
            "type" => "e-commerce",
            "operation_id" => $idOperation,
            "signature" => $signature
        ]);

        return $this->frakmenta_api_connection(Configuration::get('FRAKMENTA_URL').'/api/fk/v2/operations/status', $data, 'POST');
    }

    private function sendEmail($message)
    {
        Mail::Send(
            (int)(Configuration::get('PS_LANG_DEFAULT')), // defaut language id
            'contact', // email template file to be use
            ' Error en transaccion fk prestashop - '._PS_VERSION_, // email subject
            array(
                '{email}' => Configuration::get('PS_SHOP_EMAIL'), // sender email address
                '{message}' => $message // email content
            ),
            'desarrollo-frakmenta@findirect.com', // receiver email address
            NULL, //receiver name
            NULL, //from email address
            NULL  //from name
        );
    }
}