<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

// Cargar autoloader del módulo
require_once dirname(__FILE__) . '/../../autoload.php';

use Frakmenta\Service\FrakmentaApiClient;
use Frakmenta\Service\FrakmentaTransactionService;
use Frakmenta\Service\FrakmentaConfigService;
use Frakmenta\Repository\FrakmentaTransactionRepository;
use Frakmenta\Exception\FrakmentaApiException;

/**
 * Class FrakmentaPaymentModuleFrontController
 * 
 * Controlador para manejar la creación de operaciones de pago
 * Compatible con PHP 8.4
 */
class FrakmentaPaymentModuleFrontController extends ModuleFrontController
{
    /**
     * @var bool Forzar SSL
     */
    public $ssl = true;
    
    /**
     * @var bool Ocultar columna izquierda
     */
    public $display_column_left = false;

    /**
     * Inicializa el contenido del controlador
     * Crea una operación de pago y devuelve la URL del token
     *
     * @return void
     */
    public function initContent()
    {
        parent::initContent();
        $this->ajax = true;
        
        // Log inicial
        if (_PS_MODE_DEV_) {
            PrestaShopLogger::addLog(
                'Frakmenta: Iniciando proceso de pago - Cart ID: ' . ($this->context->cart->id ?? 'null'),
                1,
                null,
                'Cart',
                $this->context->cart->id ?? 0
            );
        }
        
        try {
            $cart = $this->context->cart;
            
            // Validar que el carrito tenga contenido
            if (!$this->validateCart($cart)) {
                PrestaShopLogger::addLog('Frakmenta: Carrito inválido o vacío', 2);
                $this->ajaxRender(json_encode(['error' => 'Invalid cart', 'status' => 'error']));
                return;
            }
            
            if (_PS_MODE_DEV_) {
                PrestaShopLogger::addLog(
                    'Frakmenta: Carrito válido - Total: ' . $cart->getOrderTotal(true, Cart::BOTH),
                    1
                );
            }
            
            // Crear servicios
            $apiClient = new FrakmentaApiClient(
                FrakmentaConfigService::getApiUrl(),
                FrakmentaConfigService::getPublicKey(),
                FrakmentaConfigService::getPrivateKey()
            );
            
            if (_PS_MODE_DEV_) {
                PrestaShopLogger::addLog(
                    'Frakmenta: ApiClient creado - URL: ' . FrakmentaConfigService::getApiUrl(),
                    1
                );
            }
            
            $repository = new FrakmentaTransactionRepository();
            $transactionService = new FrakmentaTransactionService(
                $apiClient,
                $repository,
                FrakmentaConfigService::getMerchantId()
            );
            
            if (_PS_MODE_DEV_) {
                PrestaShopLogger::addLog(
                    'Frakmenta: TransactionService creado - Merchant ID: ' . FrakmentaConfigService::getMerchantId(),
                    1
                );
            }
            
            // Crear operación
            if (_PS_MODE_DEV_) {
                PrestaShopLogger::addLog('Frakmenta: Llamando a createOperation...', 1);
            }
            $response = $transactionService->createOperation($cart, $this->context->customer);
            
            if (_PS_MODE_DEV_) {
                PrestaShopLogger::addLog(
                    'Frakmenta: Respuesta recibida - Status: ' . ($response['status'] ?? 'unknown'),
                    1
                );
            }

            // Verificar respuesta
            if (strtoupper($response['status']) !== 'OK') {
                PrestaShopLogger::addLog(
                    'Frakmenta: Operación falló - Response: ' . print_r($response, true),
                    4
                );
                $this->ajaxRender(json_encode(['error' => 'Operation failed', 'status' => 'error']));
                return;
            }
            
            if (_PS_MODE_DEV_) {
                PrestaShopLogger::addLog(
                    'Frakmenta: Operación exitosa - Token URL: ' . ($response['data']['token_url'] ?? 'missing'),
                    4
                );
            }
            
            // Devolver URL del token
            $this->ajaxRender($response['data']['token_url']);
            
        } catch (FrakmentaApiException $e) {
            PrestaShopLogger::addLog(
                'Frakmenta: FrakmentaApiException - ' . $e->getMessage() . ' | Code: ' . $e->getHttpCode(),
                3,
                null,
                'Cart',
                $this->context->cart->id ?? 0
            );
            $this->logError($e);
            $this->ajaxRender(json_encode([
                'error' => $e->getMessage(),
                'status' => 'error',
                'http_code' => $e->getHttpCode(),
                'details' => $response ?? null
            ]));
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                'Frakmenta: Exception inesperada - ' . $e->getMessage() . ' | File: ' . $e->getFile() . ' | Line: ' . $e->getLine(),
                3,
                null,
                'Cart',
                $this->context->cart->id ?? 0
            );
            $this->logError($e);
            $this->ajaxRender(json_encode([
                'error' => 'Unexpected error: ' . $e->getMessage(),
                'status' => 'error',
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'details' => $response ?? null
            ]));
        }
    }
    
    /**
     * Valida que el carrito sea válido
     *
     * @param Cart $cart Carrito a validar
     * @return bool True si el carrito es válido
     */
    private function validateCart(Cart $cart): bool
    {
        return $cart->getOrderTotal(true, Cart::BOTH) > 0;
    }
    
    /**
     * Registra un error en el log
     *
     * @param Exception $e Excepción a registrar
     * @return void
     */
    private function logError(Exception $e): void
    {
        PrestaShopLogger::addLog(
            sprintf('Frakmenta payment error: %s', $e->getMessage()),
            3,
            null,
            'Cart',
            $this->context->cart->id ?? 0
        );
    }
}
