<?php
/**
 * 2021 Sistemas findirect
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 *
 * @author Jose Baez
 * @author <desarrollo@frakmenta.com>
 * @copyright Sistemas findirect
 * @license http://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

// Cargar autoloader del módulo
require_once dirname(__FILE__) . '/../../autoload.php';

use Frakmenta\Repository\FrakmentaTransactionRepository;
use Frakmenta\Config\FrakmentaConstants;

/**
 * Class FrakmentaNotificationsModuleFrontController
 * 
 * Controlador para recibir notificaciones de Frakmenta (webhooks)
 * Compatible con PHP 8.4
 */
class FrakmentaNotificationsModuleFrontController extends ModuleFrontController
{
    /**
     * @var bool Forzar SSL
     */
    public $ssl = true;
    
    /**
     * @var bool Ocultar columna izquierda
     */
    public $display_column_left = false;

    /**
     * Procesa las notificaciones de Frakmenta
     *
     * @return void
     */
    public function initContent()
    {
        parent::initContent();

        try {
            // Obtener datos de la notificación
            $rawInput = file_get_contents('php://input');
            
            if ($rawInput === false || empty($rawInput)) {
                $this->logError('Empty notification received');
                http_response_code(400);
                $this->setTemplate('module:frakmenta/views/templates/front/unknow.tpl', []);
                return;
            }
            
            $notification = json_decode($rawInput, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                $this->logError('JSON decode error: ' . json_last_error_msg());
                http_response_code(400);
                $this->setTemplate('module:frakmenta/views/templates/front/unknow.tpl', []);
                return;
            }
            
            // Validar estructura de la notificación
            if (!isset($notification['data']['invoice_id']) || !isset($notification['status'])) {
                $this->logError('Invalid notification structure');
                http_response_code(400);
                $this->setTemplate('module:frakmenta/views/templates/front/unknow.tpl', []);
                return;
            }
            
            // Procesar notificación
            $template = $this->processNotification($notification);
            $this->setTemplate($template, []);
            
        } catch (Exception $e) {
            $this->logError('Notification processing error: ' . $e->getMessage());
            http_response_code(500);
            $this->setTemplate('module:frakmenta/views/templates/front/unknow.tpl', []);
        }
    }
    
    /**
     * Procesa una notificación de Frakmenta
     *
     * @param array $notification Datos de la notificación
     * @return string Ruta de la plantilla a mostrar
     */
    private function processNotification(array $notification): string
    {
        $repository = new FrakmentaTransactionRepository();
        $fk = new Frakmenta();
        
        $invoiceId = $notification['data']['invoice_id'];
        $status = $notification['status'];
        
        // Buscar transacción
        $result = $repository->findAllByInvoiceId($invoiceId);
        
        if (!$result) {
            $this->logError('Transaction not found: ' . $invoiceId);
            return 'module:frakmenta/views/templates/front/unknow.tpl';
        }
        
        // Si está inicializada, procesar
        if ($result[0]['petition_state'] === FrakmentaConstants::STATE_INITIALIZED) {
            $orderId = (int) $fk->frakmentaManagerTransaction($result, $status);
            
            if (!empty($orderId)) {
                $message = $notification['messages'][0] ?? '';
                $repository->update($invoiceId, [
                    'petition_state' => $status,
                    'state_detail' => $message,
                    'id_order' => $orderId
                ]);
                
                return 'module:frakmenta/views/templates/front/received.tpl';
            }
        } elseif ($result[0]['state_detail'] === 'La operación ha sido aceptada') {
            return 'module:frakmenta/views/templates/front/received.tpl';
        }
        
        return 'module:frakmenta/views/templates/front/unknow.tpl';
    }
    
    /**
     * Registra un error en el log
     *
     * @param string $message Mensaje de error
     * @return void
     */
    private function logError(string $message): void
    {
        PrestaShopLogger::addLog(
            'Frakmenta notification: ' . $message,
            3,
            null,
            'Frakmenta',
            0
        );
    }
}